import xml.etree.ElementTree as ET
import csv
import argparse
import logging
from datetime import datetime
import os

def argument_parser():
    parser = argparse.ArgumentParser(
        description="Create a CSV report on all IN Events with no OUT")
    parser.add_argument('-i', dest='input_file', type=str, default='',required=True,
                        help='Input XML file', metavar='')
    parser.add_argument('-o', dest='output_file', type=str,default='',required=True,
                        help='Output CSV file', metavar='')
    parser.add_argument('-log', dest='log', type=int,
                        help=argparse.SUPPRESS, default=0, metavar='')
    return parser

def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info(get_time_stamp() + ' script parameters: ' + str(args))

    tree = ET.parse(args.input_file)
    root = tree.getroot()

    with open(args.output_file, 'w', newline='') as csv_file:
        csv_writer = csv.writer(csv_file)
        csv_writer.writerow(['Folder Name', 'Job name', 'Event with no OUT'])
        in_events_dict = get_all_in_events(root)
        list_all_in_events_without_out(root, in_events_dict, csv_writer)
        
def get_all_in_events(root):
    in_events_dict = get_in_events_from_folders(root, "SMART_FOLDER")
    in_events_dict = in_events_dict | get_in_events_from_folders(root, "FOLDER")
    return in_events_dict
    
def get_in_events_from_folders(root, folder_type):
    in_events_dict: dict = {}
    for folder in root.iter(folder_type):
        folder_name = folder.attrib.get("FOLDER_NAME", "")
        for job in folder.iter("JOB"):
            for cond in job.iter("INCOND"):
                in_events_dict.update({cond.attrib["NAME"] : folder_job(get_jobname(job), folder_name)})
                logging.debug(get_time_stamp() + 'event {} found in folder {} job {}'.format(cond.attrib["NAME"], folder_name, get_jobname(job)) )
    return in_events_dict

def list_all_in_events_without_out(root, in_events_dict, csv_writer):
    logging.info(get_time_stamp() + 'Seraching for matching out_cond' )
    remove_out_events_of_folders(root, in_events_dict, "SMART_FOLDER")
    remove_out_events_of_folders(root, in_events_dict, "FOLDER")
    report_in_events_without_out(in_events_dict, csv_writer)
    return

def remove_out_events_of_folders(root, in_events_dict, folder_type):
    for folder in root.iter(folder_type):
        for job in folder.iter("JOB"):
            for cond in job.iter("OUTCOND"):
                if cond.attrib["SIGN"] == "+" or cond.attrib["SIGN"] == "ADD":
                    in_events_dict.pop(cond.attrib["NAME"], None)
                    logging.debug(get_time_stamp() + 'event {} found and removed'.format(cond.attrib["NAME"]) )

    return

def report_in_events_without_out(in_events_dict, csv_writer):
    for in_event in in_events_dict.keys():
        in_event_value = in_events_dict[in_event]
        csv_writer.writerow([in_event_value.folder_name, in_event_value.jobname, in_event])
    return

class folder_job:
  def __init__(self, jobname, folder_name):
    self.jobname = jobname
    self.folder_name = folder_name

def get_jobname(job):
    job_name = ''
    try:
        job_name = job.attrib['JOBNAME']
    except KeyError:
        job_name = job.attrib['MEMNAME']
    return job_name

def generate_log_file(log):
    dirpath = "log"
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "xml_to_csv.log")
    if os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.INFO)
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.ERROR)
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.DEBUG)

def get_time_stamp():
    return str(datetime.now()) + " "

if __name__ == "__main__":
    main()
    