import xml.etree.ElementTree as ET
import csv
import argparse
import logging
from datetime import datetime
import os

def argument_parser():
    parser = argparse.ArgumentParser(
        description="Extract folder information from XML to CSV")
    parser.add_argument('-i', dest='input_file', type=str,
                        help='Input XML file', metavar='', required=True)
    parser.add_argument('-o', dest='output_file', type=str,
                        help='Output CSV file', metavar='', required=True)
    parser.add_argument('-log', dest='log', type=int,
                        help=argparse.SUPPRESS, default=0, metavar='')
    parser.add_argument('-a', dest='folder_attr', type=str,
                        help='Folder attribute to be added to output', metavar='', required=False)                    
    parser.add_argument('-v', dest='variable_name_to_filter', type=str,
                        help='variable name to report on', metavar='', required=False)                    
    return parser

def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info(get_time_stamp() + ' script parameters: ' + str(args))

    tree = ET.parse(args.input_file)
    root = tree.getroot()

    with open(args.output_file, 'w', newline='') as csv_file:
        csv_writer = csv.writer(csv_file)
        csv_fields = ['Folder Name', 'Number of Jobs', 'Platform', 'Scheduled', 'Order Method']
        csv_fields.append(args.variable_name_to_filter) if args.variable_name_to_filter else None
        if args.folder_attr:
            csv_fields.append(args.folder_attr)
        csv_writer.writerow(csv_fields)

        for folder in root.iter("SMART_FOLDER"):
            folder_name = folder.attrib.get("FOLDER_NAME", "")
            order_method = folder.attrib.get("FOLDER_ORDER_METHOD", "")
            num_jobs = str(len(list(folder.iter("JOB"))))
            platform = folder.attrib.get("PLATFORM", "")
            scheduled = 'N'
            folder_attribute = folder.attrib.get(args.folder_attr, "")
            variable_value = ''
            for calendar in folder.findall('.//RULE_BASED_CALENDAR'):
                dayscal = calendar.get('DAYSCAL')
                weekscal = calendar.get('WEEKSCAL')
                date = calendar.get('DATE')
                weekdays = calendar.get('WEEKDAYS')
                days = calendar.get('DAYS')
                if dayscal is not None or weekscal is not None or date is not None or weekdays is not None or days is not None:
                    scheduled = 'Y' 
                    break
            if args.variable_name_to_filter is not None:        
                variable_name = args.variable_name_to_filter 
                xpath_expression = f'.//VARIABLE[@NAME="%%{variable_name}"]'
                variable_element = folder.find(xpath_expression)
                if variable_element is None:
                   variable_value = 'N'
                else:
                    variable_value = variable_element.get('VALUE')
                    
            row = [folder_name, num_jobs, platform, scheduled, order_method]
            if args.variable_name_to_filter:
                row.append(variable_value)
            if args.folder_attr:
                row.append(folder_attribute)
            csv_writer.writerow(row)

    print(f"Data extracted from {args.input_file} and saved to {args.output_file}")

def generate_log_file(log):
    dirpath = "log"
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "xml_to_csv.log")
    if os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.INFO)
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.ERROR)
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.DEBUG)

def get_time_stamp():
    return str(datetime.now()) + " "

if __name__ == "__main__":
    main()
    