import datetime
import os

try:
    from openpyxl import Workbook
except ImportError:
    from pip._internal import main as pip

    pip(['install', '--user', 'openpyxl'])
    from openpyxl import Workbook

from openpyxl.styles import Font, PatternFill, Alignment

from config_parms import config_parms

color = '808080'


class write_results:
    #
    # This class handles the creation of the excel file with the results.
    #
    def __init__(self, cp: config_parms):
        self._wb_ = Workbook()
        self._parm_ = cp

    def write_to_file(self, grp_list, jinfo=None):
        #
        # This is the main method that writes the entire excel file with the results
        #

        os.makedirs(os.path.dirname(self._parm_.get_output_folder()), exist_ok=True)
        output_file_name = self._parm_.get_output_folder() + '/output.xlsx'

        print(' ' * 200, end='\r')  # clear for spinner message...
        print('Writing output to ' + output_file_name + '...', end='\r')  # Spinner message...

        self._write_summary_(grp_list, jinfo)  # Print the summary sheet

        for group in grp_list:  # Print the group sheets (one sheet per group)
            self._write_group_(group)

        self._wb_.save(output_file_name)  # Save the excel file

        print('Number of groups generated: ' + str(len(grp_list)) + '. Output written to ' + output_file_name)

    def _write_summary_(self, grp_list, jinfo):
        #
        # Prints the summary sheet (total execution info + config parms details)
        #
        total_jobs = 0
        total_folders = 0

        for g in grp_list:
            total_jobs = total_jobs + g.jobs_count()
            total_folders = total_folders + g.folders_count()

        ws = self._wb_.active
        ws.title = 'Summary and Config'

        ws.column_dimensions['A'].width = 50
        ws.column_dimensions['B'].width = 30
        ws.column_dimensions['C'].width = 30

        ws['A1'] = 'Execution Completed on ' + str(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S') + ':')
        ws['A1'].font = Font(bold=True, underline="single")
        ws['B1'] = str(total_folders) + ' folders with ' + str(
            total_jobs) + ' jobs were processed and splitted to ' + str(len(grp_list)) + ' groups.'
        ws['B1'].font = Font(bold=True, underline="single")
        ws['B1'].alignment = Alignment(wrap_text=True)

        current_row = 3  # Begin from row 3
        current_row = self._write_groups_summary_(ws, current_row, grp_list)  # Write groups summary
        current_row = self._write_config_parms_(ws, current_row + 1)  # Write the config parms

        if self._parm_.generate_sub_jsons() and int(self._parm_.max_jobs_in_json()) > 0:
            self._write_jsons_files_info_(ws, jinfo)

    def _write_jsons_files_info_(self, ws, jinfo):
        ws.column_dimensions['E'].width = 2
        ws.column_dimensions['F'].width = 30
        ws.column_dimensions['G'].width = 30

        ws.cell(row=2, column=5).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=3, column=5).fill = PatternFill(patternType='solid', bgColor=color)
        ws['F3'] = 'Jsons generated:'
        ws['F3'].font = Font(bold=True, underline="single")

        ws.cell(row=4, column=5).fill = PatternFill(patternType='solid', bgColor=color)
        ws['F4'] = 'Json name'
        ws['F4'].font = Font(bold=True, underline="single")
        ws['G4'] = 'Jobs in json'
        ws['G4'].font = Font(bold=True, underline="single")
        ws.cell(row=3, column=5).fill = PatternFill(patternType='solid', bgColor=color)

        r = 5
        for j in jinfo:
            ws.cell(row=r, column=5).fill = PatternFill(patternType='solid', bgColor=color)
            ws.cell(row=r + 1, column=5).fill = PatternFill(patternType='solid', bgColor=color)
            ws.cell(row=r, column=6).value = j + '.json'
            ws.cell(row=r, column=7).value = jinfo[j]
            if jinfo[j] > int(self._parm_.max_jobs_in_json()):
                ft = Font(color="FF0000")
                ws.cell(row=r, column=7).font = ft
            r = r + 1

    def _write_groups_summary_(self, ws, r, grp_list):
        #
        # Writes the groups summary lines in the summary sheet
        #
        ws.cell(row=r, column=1).value = 'Groups Generated:'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")
        r = r + 1

        ws.cell(row=r, column=1).value = 'Group Name'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")

        ws.cell(row=r, column=2).value = 'Jobs Counter'
        ws.cell(row=r, column=2).font = Font(bold=True, underline="single")

        ws.cell(row=r, column=3).value = 'Folders Counter'
        ws.cell(row=r, column=3).font = Font(bold=True, underline="single")

        if self._parm_.generate_sub_jsons():
            ws.cell(row=r, column=4).value = 'Output json file name'
            ws.cell(row=r, column=4).font = Font(bold=True, underline="single")
            ws.column_dimensions['D'].width = 30

        for g in grp_list:
            r = r + 1
            ws.cell(row=r, column=1).value = g.group_name()
            ws.cell(row=r, column=2).value = g.jobs_count()
            ws.cell(row=r, column=3).value = g.folders_count()
            if self._parm_.generate_sub_jsons():
                ws.cell(row=r, column=4).value = g.json_name()

        return (r + 1)

    def _write_config_parms_(self, ws, r):
        #
        # Writes the configuration parameters in the summary sheet
        #
        ws.cell(row=r, column=1).value = 'Configuration Parameters:'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")
        r = r + 1

        ws.cell(row=r, column=1).value = 'input_file_name'
        ws.cell(row=r, column=2).value = self._parm_.get_input_file_name()
        r = r + 1

        ws.cell(row=r, column=1).value = 'output_folder'
        ws.cell(row=r, column=2).value = self._parm_.get_output_folder()
        r = r + 1

        ws.cell(row=r, column=1).value = 'manage_events'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.manage_events())
        r = r + 1

        ws.cell(row=r, column=1).value = 'manage_vars'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.manage_vars())
        r = r + 1

        ws.cell(row=r, column=1).value = 'manage_res'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.manage_res())
        r = r + 1

        ws.cell(row=r, column=1).value = 'manage_qres'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.manage_qres())
        r = r + 1

        ws.cell(row=r, column=1).value = 'generate_sub_jsons'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.generate_sub_jsons())
        r = r + 1

        ws.cell(row=r, column=1).value = 'max_jobs_in_json'
        ws.cell(row=r, column=2).value = str(self._parm_.max_jobs_in_json())
        r = r + 1

        ws.cell(row=r, column=1).value = 'write_output_to_terminal'
        ws.cell(row=r, column=2).value = self._bool_to_y_n_(self._parm_.write_output_to_terminal())
        r = r + 2

        ws.cell(row=r, column=1).value = 'Exclude Resource List:'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")

        excList = self._parm_.get_exclude_list()
        for item in excList:
            r = r + 1
            ws.cell(row=r, column=1).value = item

        return (r + 1)

    def _bool_to_y_n_(self, b):
        #
        # Convert bool value to Yes/No string
        #
        if b:
            return ('Yes')
        else:
            return ('No')

    def _write_group_(self, group):
        #
        # Write a group sheet for a given group
        #
        ws = self._wb_.create_sheet(group.group_name())  # Create a new sheet for the group

        ws.column_dimensions['A'].width = 80
        ws.column_dimensions['B'].width = 30
        ws.column_dimensions['C'].width = 2
        ws.column_dimensions['D'].width = 80
        ws.column_dimensions['E'].width = 30

        ws['A1'] = 'Group Name: ' + group.group_name()  # Print the title
        ws['A1'].font = Font(bold=True, underline="single")

        self._write_group_summary_(ws, group)  # Print the groups summary lines
        self._write_group_folders_(ws, group, 9)  # Print the group folders
        self._write_group_resources_(ws, group, 9)  # Print the group resources

    def _write_group_summary_(self, ws, group):
        #
        # Print the group summary lines in the group sheet
        #
        ws['A3'] = 'Total Folders in Group: '
        ws['B3'] = group.folders_count()

        ws['A4'] = 'Total Jobs in Group: '
        ws['B4'] = group.jobs_count()

        if self._parm_.generate_sub_jsons():
            ws['A5'] = 'Included in output json file: '
            ws['B5'] = group.json_name()

        ws.cell(row=7, column=1).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=7, column=2).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=7, column=3).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=7, column=4).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=7, column=5).fill = PatternFill(patternType='solid', bgColor=color)

    def _write_group_folders_(self, ws, group, r):
        #
        # Print the group folders in the group sheet
        #
        ws.cell(row=r, column=1).value = 'Folders In this Group:'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")
        ws.cell(row=r - 1, column=3).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=r, column=3).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=r + 1, column=3).fill = PatternFill(patternType='solid', bgColor=color)
        ws.cell(row=r + 2, column=3).fill = PatternFill(patternType='solid', bgColor=color)
        r = r + 2

        ws.cell(row=r, column=1).value = 'Folder Name'
        ws.cell(row=r, column=1).font = Font(bold=True, underline="single")

        ws.cell(row=r, column=2).value = 'Number Of Jobs In Folder'
        ws.cell(row=r, column=2).font = Font(bold=True, underline="single")
        r = r + 1

        folder_sorted = sorted(group.folder_names(), key=lambda x: x[1], reverse=True)
        for f in folder_sorted:
            ws.cell(row=r, column=1).value = f[0]
            ws.cell(row=r, column=2).value = f[1]
            ws.cell(row=r, column=3).fill = PatternFill(patternType='solid', bgColor=color)
            r = r + 1

    def _write_group_resources_(self, ws, group, r):
        #
        # Print the group resources in the group sheet (only the ones used by more than 1 job)
        #
        ws.cell(row=r, column=4).value = 'Common Objects Used By More Than one folder In This Group:'
        ws.cell(row=r, column=4).font = Font(bold=True, underline="single")
        r = r + 2

        ws.cell(row=r, column=4).value = 'Object Name'
        ws.cell(row=r, column=4).font = Font(bold=True, underline="single")

        ws.cell(row=r, column=5).value = 'Folders Use Count'
        ws.cell(row=r, column=5).font = Font(bold=True, underline="single")
        r = r + 1

        res_sorted = sorted(group.resources(), key=group.resources().get, reverse=True)
        for res in res_sorted:
            res_cnt = group.resources()[res]
            if res_cnt > 1:
                ws.cell(row=r, column=3).fill = PatternFill(patternType='solid', bgColor=color)
                ws.cell(row=r, column=4).value = res
                ws.cell(row=r, column=5).value = res_cnt
                r = r + 1
            else:
                break  # the resources are sorted by res_cnt so no need to continue the loop
