#
# Folder Group Class
# ------------------
#
# This class represents a group of folders that share at least one common resource with at least one other folder in the group.  
# The group has the following attributes: 
#   + Its name (auto generated or given - helps in debug...)
#   + List of folder names in the group (and the number of jobs in each) 
#   + General jobs counter (accumulation of the jobs number from all the folders) 
#   + Collection of "resources" used by the folders in the group (and the usage counter)
#   + Indication that is used by the merge process when a common resource is found and the group needs to be merged with another. 
#
from folder import folder


class folder_group:
    _group_name_counter_ = 0

    def __init__(self, f: folder):
        folder_group._group_name_counter_ = folder_group._group_name_counter_ + 1
        self._group_name_ = 'Group_' + str(folder_group._group_name_counter_).zfill(10)
        self._folder_names_ = [[f.folder_name(), f.jobs_counter()]]
        self._folder_seqs_ = [f.folder_seq()]
        self._jobs_counter_ = f.jobs_counter()
        self._group_resources_ = dict()
        self._merge_resources_(f.resources())

        self._json_name_ = None
        self._need_merge_ = False

    def add_folder(self, f: folder):
        #
        # Add a folder to the group (add its name to the folder's list, add its jobs count to the group counter and merge the folder resources with the group resources)
        #
        self._jobs_counter_ = self._jobs_counter_ + f.jobs_counter()
        self._folder_names_.append([f.folder_name(), f.jobs_counter()])
        self._folder_seqs_.append(f.folder_seq())
        self._merge_resources_(f.resources())

    def _merge_resources_(self, d: dict, group_merge=False):
        #
        # Merge resources dictionary with anbother (Add the resources that do not already exist or adjust the use count if already exist)
        # When invoked with group_merge=True then the merge is between groups resources (and not folders) so the use counters need to be accumulated.
        #
        for key in d:
            if key in self._group_resources_:
                if group_merge:
                    self._group_resources_[key] = self._group_resources_[key] + d[key]
                else:
                    self._group_resources_[key] = self._group_resources_[key] + 1
            else:
                if group_merge:
                    self._group_resources_[key] = d[key]
                else:
                    self._group_resources_[key] = 1

    def group_name(self):
        return self._group_name_

    def set_group_name(self, name):
        self._group_name_ = name

    def folders_count(self):
        return len(self._folder_names_)

    def folder_names(self):
        return self._folder_names_

    def folder_seqs(self):
        return self._folder_seqs_

    def json_name(self):
        if self._json_name_ == None:
            return self._group_name_ + '.json'
        else:
            return self._json_name_ + '.json'

    def set_json_name(self, jname):
        self._json_name_ = jname

    def jobs_count(self):
        return self._jobs_counter_

    def resources(self):
        return self._group_resources_

    def need_merge(self):
        return (self._need_merge_)

    def set_need_merge(self, s: bool):
        self._need_merge_ = s

    def group_info(self):
        print('Group name: ', self._group_name_)
        print('Group folders counter: ', len(self._folder_names_))
        print('Folders: ', self._folder_names_)
        print('Jobs counter: ', self._jobs_counter_)
        print('Resources counter: ', len(self._group_resources_))
        print('Resources: ', self._group_resources_)

    def merge_with(self, g):
        #
        # Merge two groups into one
        #
        self._folder_names_ = self._folder_names_ + g.folder_names()
        self._folder_seqs_ = self._folder_seqs_ + g.folder_seqs()
        self._jobs_counter_ = self._jobs_counter_ + g.jobs_count()
        self._merge_resources_(g.resources(), group_merge=True)

    def res_intersects_with(self, f):
        #
        # Check if one group intersects with a folder (i.e. if they have at least one resouce in common)
        #
        for key in f.resources():
            if key in self._group_resources_:
                return True
        return False
