class config_parms:
    #
    # Manage the configuration file.
    # This class reads the configuration file (only on the 1st object instance), sets the parameters values and populates the exclude list if an exclude file is specified.
    #

    _already_initialized_ = False
    _input_file_name_ = 'input.json'
    _output_folder_ = '/output'
    _manage_events_ = True
    _manage_vars_ = True
    _manage_res_ = True
    _manage_qres_ = True
    _generate_sub_jsons_ = True
    _max_jobs_in_json_ = 0
    _exclude_list_ = []
    _write_output_to_terminal_ = True

    def __init__(self, config_file_name='dummy'):
        #
        # Initialize the configuration patameters only once in case several instances of this object are created
        #
        if config_parms._already_initialized_ == False and config_file_name != 'dummy':
            self._read_config_(config_file_name)
            config_parms._already_initialized_ = True

    def _read_config_(self, config_file_name):
        #
        # Read the configuration file: extract parm and values pairs and validate them
        #
        with open(config_file_name, 'r') as f:

            line = f.readline()
            while line:
                parm_pair = self._validate_line(line, config_file_name)

                if parm_pair != False:
                    parm_name = parm_pair[0]
                    parm_value = parm_pair[1]
                    self._manage_parm_(parm_name, parm_value, config_file_name)

                line = f.readline()

    err_line_header = 'Line ignored in parameters file '

    def _validate_line(self, line, config_file_name):
        #
        # Validate a line read from the config file and make sure it is in a parm=val format
        #
        if line[0:1] != '#' and len(line.strip()) > 0:
            line = line.strip()
            epos = line.find('=')
            if epos >= 0:
                parm_name = line[0:epos].strip()
                parm_value = line[epos + 1:len(line)].strip()
                return ([parm_name, parm_value])
            else:
                print(config_parms.err_line_header + config_file_name + ' - invlaid format: ' + line)
        return False

    def _manage_parm_(self, name, value, config_file_name):
        #
        # Check if the pair of parm=value are known parameters
        #
        match name.upper():

            case 'INPUT_FILE_NAME':
                config_parms._input_file_name_ = value
                return ()

            case 'OUTPUT_FOLDER':
                config_parms._output_folder_ = value
                return ()

            case 'MANAGE_EVENTS':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val is not None:
                    config_parms._manage_events_ = ret_val
                return ()

            case 'MANAGE_VARS':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val is not None:
                    config_parms._manage_vars_ = ret_val
                return ()

            case 'MANAGE_RES':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val is not None:
                    config_parms._manage_res_ = ret_val
                return ()

            case 'MANAGE_QRES':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val is not None:
                    config_parms._manage_qres_ = ret_val
                return ()

            case 'GENERATE_SUB_JSONS':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val is not None:
                    config_parms._generate_sub_jsons_ = ret_val
                return ()

            case 'MAX_JOBS_IN_JSON':
                if value.isnumeric():
                    config_parms._max_jobs_in_json_ = value
                else:
                    config_parms._max_jobs_in_json_ = 0
                    print(config_parms.err_line_header + config_file_name + ' - Numeric value >0 expected in ' + name)
                return ()

            case 'EXCLUDE_FILE_NAME':
                self._load_exclude_list_(value)
                return ()

            case 'WRITE_OUTPUT_TO_TERMINAL':
                ret_val = self._check_yes_no_(name, value, config_file_name)
                if ret_val != None:
                    config_parms._write_output_to_terminal_ = ret_val
                return ()

        print(config_parms.err_line_header + config_file_name + ' - Unknown parameter: ' + name)

    def _check_yes_no_(self, name, value, config_file_name):
        #
        # Check if the value of a Y/N parameter is valid
        #
        if value.upper() == 'Y' or value.upper() == 'YES':
            return (True)

        if value.upper() == 'N' or value.upper() == 'NO':
            return (False)

        if len(value.strip()) > 0:
            print(
                config_parms.err_line_header + config_file_name + ' - Invalid value ' + value + ' for parameter ' + name)
        else:
            print(config_parms.err_line_header + config_file_name + ' - No value for parameter ' + name)
        return (None)

    def validate_file_or_exit(self, file_name, message):
        def _validate_file_exists():
            try:
                with open(file_name, 'r'):
                    return True
            except FileNotFoundError:
                return False

        if not _validate_file_exists():
                print(message)
                exit(1)
        return file_name



    def _load_exclude_list_(self, exclude_file_name):
        #
        # Load the exclude file into the exclude resources list
        #
        self.validate_file_or_exit(exclude_file_name, f"Warning: The exclude file '{exclude_file_name}' could not be found. Please verify that the file path is correct and that the file exists.")
        with open(exclude_file_name, 'r') as f:
            line = f.readline()
            while line:
                if line[0:1] != '#' and len(line.strip()) > 0:
                    config_parms._exclude_list_.append(line.strip())
                line = f.readline()

    #
    # Return parms values methods
    #

    def get_input_file_name(self):
        filename = config_parms._input_file_name_
        parameter_name= 'input_file_name'
        return self.validate_file_or_exit(filename,
                                     f"Warning: The file specified in the parameter '{parameter_name}' ('{filename}') could not be found. Please verify that the file path is correct and that the file exists.")


    def get_output_folder(self):
        return config_parms._output_folder_

    def manage_events(self):
        return config_parms._manage_events_

    def manage_vars(self):
        return config_parms._manage_vars_

    def manage_res(self):
        return config_parms._manage_res_

    def manage_qres(self):
        return config_parms._manage_qres_

    def generate_sub_jsons(self):
        return config_parms._generate_sub_jsons_

    def max_jobs_in_json(self):
        return config_parms._max_jobs_in_json_

    def get_exclude_list(self):
        return config_parms._exclude_list_

    def write_output_to_terminal(self):
        return config_parms._write_output_to_terminal_

    def print_config_parms(self):
        #
        # Print the config parms values (debug purposes)
        #
        print('Configuration Setting:')
        print('---------------------')
        print('input_file_name = ' + config_parms._input_file_name_)
        print('output_folder = ' + config_parms._output_folder_)
        print('manage_events = ' + str(config_parms._manage_events_))
        print('manage_vars = ' + str(config_parms._manage_vars_))
        print('manage_res = ' + str(config_parms._manage_res_))
        print('manage_qres = ' + str(config_parms._manage_qres_))
        print('generate_sub_jsons = ' + str(config_parms._generate_sub_jsons_))
        print('write_output_to_terminal = ' + str(config_parms._write_output_to_terminal_))
        print('exclude resource list:')
        for item in config_parms._exclude_list_:
            print(' - ' + item)
        print('')
