#!/usr/bin/python
#coding=utf8

import html
import re
from xml.dom import minidom
import xml.etree.cElementTree as cET			
import xml.etree.ElementTree as ET			
import os, time, math		
import argparse
import logging
import datetime
from datetime import datetime

def argument_parser():
    parser = argparse.ArgumentParser(
       description=USAGE)
    parser.add_argument('-i', dest='input_file', type=str, default='in.xml', help='Input XML file with TWS plugin jobs', metavar='')
    parser.add_argument('-o', dest='output_file', type=str, help='Output XML file where supported TWS plugin jobs are converted to their Control-M equivalents', metavar='', default='out.xml')
    parser.add_argument('-log', dest='log', type=int, help='log level - file will be created in the directory where this script resides', default=0, metavar='')
    return parser

USAGE= "jdsl-converter.py -i <inputfile> -o <outputfile>"
def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info('script parameters: ' + str(args))

    print ('Input is: ' + args.input_file)  
    print ('Output is: ' + args.output_file)

    global var_dictionary
    var_dictionary = {
    "OXJOBNAM": "JOBNAME"
    }

    tree = cET.parse(args.input_file); print("Read the input file: " + args.input_file)
    root = tree.getroot()
    
    process_all_jobs(root, args)

    print("Writing updates to the XML file...")

    tree.write(args.output_file);print("Updates saved to the output XML file.")

def process_all_jobs(root, args):
    for job in root.iter('JOB'):
        logging.debug('Processing job {} '.format(job.attrib['JOBNAME']))
        try:
            if (job.attrib['TASKTYPE'] == 'Job' and job.attrib["JOBNAME"] and job.attrib['MEMNAME'] == 'XMLPLGIN'):
                process_script_job(job, args)
            if (job.attrib['TASKTYPE'] == 'Job' and job.attrib['MEMNAME'] == 'JTYPEJOB'):
                process_jtype_job(job, args)
        except:
            pass
    return

def process_script_job(job, args):
    temp_xml = ''
    try:
        if job.attrib['USE_INSTREAM_JCL'] == 'Y':
            temp_xml = job.attrib['INSTREAM_JCL']
    except KeyError:
        return
    start_pos = temp_xml.find('//TASKTYPE=')+ len('//TASKTYPE=')
    end_pos = temp_xml.find('\n', start_pos) 
    plugin_type = temp_xml[start_pos:end_pos].strip()
    inner_xml = temp_xml[end_pos+1:].strip().replace('\n', '').replace('\t',' ')
    try:
        inner_tree = ET.fromstring(inner_xml)
    except ET.ParseError as e:
        logging.error('Error parsing inner XML for job {}: {}'.format(job.attrib['JOBNAME'], e))
        print('Error parsing inner XML for job {}: {}'.format(job.attrib['JOBNAME'], e))
        return

    logging.debug( 'Converting job {} with type {}'.format(job.attrib['JOBNAME'], plugin_type))

    try:
        match plugin_type:
            case 'filetransfer':
                process_ftp(inner_tree, job, args)
            case 'database':
                process_database(inner_tree, job, args)
            case 'restful':
                process_restful(inner_tree, job, args)
            case 'executable':
                process_executable(inner_tree, job, args)
            case _:
                debug_msg = 'Job {} with type {} is not supported'.format(job.attrib['JOBNAME'], plugin_type)
                logging.debug(debug_msg)
                print(debug_msg)
                return
    except Exception as e:
        print('Unexpected error processing job {} type {} in application {}: {}'.format(job.attrib['JOBNAME'], plugin_type, job.attrib['APPLICATION'], e))
        logging.error(' Unexpected error processing job {} type {} in application {}: {}'.format(job.attrib['JOBNAME'], plugin_type, job.attrib['APPLICATION'], e))
        return
    logging.info('Job {} type {} in application {} converted successfully'.format(job.attrib['JOBNAME'], plugin_type,job.attrib['APPLICATION']))
    print('Job {} type {} in application {} converted successfully'.format(job.attrib['JOBNAME'], plugin_type,job.attrib['APPLICATION']))
    return

def add_var(job, name, value):
    logging.debug('Adding variable {} with value {}'.format(name, value))
    var = cET.Element('VARIABLE')
    var.attrib['NAME'] = name
    var.attrib['VALUE'] = value
    job.append(var)

def process_ftp(inner_tree, job, args):
    #if inner_tree.attrib['protocol_ft']== 'SSH':
    #    return
    job.attrib['APPL_TYPE'] = "FILE_TRANS"
    job.attrib['APPL_FORM'] = "AFT"
    job.attrib['APPL_VER']  = "ANY"
    job.attrib['CM_VER']    = 'N/A'

    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdlfiletransfer': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdlfiletransfer',
        'jsdlfiletransferkeystorePassword': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdlfiletransferkeystorePassword',
        'jsdlfiletransferlocalPassword': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdlfiletransferlocalPassword'
    }    

    remote_host = extract_variable(inner_tree.find('.//jsdlfiletransfer:server_ft', namespaces).text)
    remote_user = extract_variable(inner_tree.find('.//jsdlfiletransfer:remoteUser', namespaces).text)
    ftp_direction = extract_variable(inner_tree.find('.//jsdlfiletransfer:direction', namespaces).text)
    src_file = extract_variable(inner_tree.find('.//jsdlfiletransfer:sourceFilesValue', namespaces).text)
    dest_file = extract_variable(inner_tree.find('.//jsdlfiletransfer:destinationPath', namespaces).text)
    file_exist = extract_variable(inner_tree.find('.//jsdlfiletransfer:fileExists', namespaces).text)
    mode = inner_tree.find('.//jsdlfiletransfer:transferMode', namespaces)
    # print (mode[0].tag)
    transfer_mode = mode[0].tag.split('}')[-1]
    add_var(job, '%%FTP-ACCOUNT', 'local+' + remote_host+"_"+remote_user)
    # add_var(job, '%%FTP-LOSTYPE', "platform") # what is that value for? let's discuss
    add_var(job, '%%FTP-CONNTYPE1','LOCAL')
    add_var(job, '%%FTP-LHOST','Local')

    add_var(job, '%%FTP-CONNTYPE2','FTP')
    add_var(job, '%%FTP-RHOST',remote_host)
    add_var(job, '%%FTP-RUSER',remote_user)
    add_var(job, '%%FTP-RPASSIVE','0')
    add_var(job, '%%FTP-CONT_EXE_NOTOK','0')
    add_var(job, '%%FTP-RPF','1')
    add_var(job, '%%FTP-USE_DEF_NUMRETRIES','0')

    add_var(job, '%%FTP-TRANSFER_NUM','1')
    add_var(job, '%%FTP-AUTOREFRESH','False')
    add_var(job, '%%FTP-Is','5')
    if transfer_mode.lower() == 'binary':
        add_var(job, '%%FTP-TYPE1','B')
    else:
        add_var(job, '%%FTP-TYPE1','I')
    if ftp_direction.lower() == 'receive':    
        add_var(job, '%%FTP-UPLOAD1','1') # 1 for download, 0 for upload
        add_var(job, '%%FTP-RPATH1',dest_file)
        add_var(job, '%%FTP-LPATH1',src_file)
    else:
        add_var(job, '%%FTP-UPLOAD1','0') # 1 for download, 0 for upload       
        add_var(job, '%%FTP-RPATH1',src_file)
        add_var(job, '%%FTP-LPATH1',dest_file)
    #add_var(job, '%%FTP-COMPRESSION11','0')
    #add_var(job, '%%FTP-COMPRESSION21','0')
    #add_var(job, '%%FTP-ABSTIME1','0')
    #add_var(job, '%%FTP-MINSIZE1','0')
    #add_var(job, '%%FTP-TIMELIMIT1','0')
    #add_var(job, '%%FTP-TIMELIMIT_UNIT1','1')
    #add_var(job, '%%FTP-OVERRIDE_WATCH_INTERVAL1','0')
    #add_var(job, '%%FTP-FW_SKIP1','0')
    #add_var(job, '%%FTP-UNIQUE1','0')
    #add_var(job, '%%FTP-SRCOPT1','0')
    if file_exist.lower() == 'replace':
        add_var(job, '%%FTP-IF_EXIST1','0')
    else: #check what options TWS supports and what is our equivalent
        add_var(job, '%%FTP-IF_EXIST1','1')
    #add_var(job, '%%FTP-DSTOPT1','0')
    #add_var(job, '%%FTP-CONT_EXE1','0')
    #add_var(job, '%%FTP-MODIFIED_SINCE_LASTRUN1','0')
    #add_var(job, '%%FTP-LAST_SUCCESFUL_RUN_INIT_AGE1','0')
    #add_var(job, '%%FTP-FIRST_RUN_DELTA1','1')
    #add_var(job, '%%FTP-POSTCMD_ON_FAILURE1','0')
    #add_var(job, '%%FTP-DEL_DEST_ON_FAILURE1','0')
    #add_var(job, '%%FTP-SYNC_DIR_NO_DEL1','0')
    #add_var(job, '%%FTP-DEST_ACT_FAIL1','0')
    #add_var(job, '%%FTP-SRC_CMD_FAIL1','0')
    #add_var(job, '%%FTP-DEST_CMD_FAIL1','0')
    #add_var(job, '%%FTP-RECURSIVE1','0')
    #add_var(job, '%%FTP-EXCLUDE_WILDCARD1','0')
    #add_var(job, '%%FTP-TRIM1','1')
    #add_var(job, '%%FTP-NULLFLDS1','0')
    #add_var(job, '%%FTP-VERNUM1','0')
    #add_var(job, '%%FTP-CASEIFS1','0')
    #add_var(job, '%%FTP-PARALLEL_TRANSFERS1','0')

    del job.attrib['MEMNAME']
    del job.attrib['MEMLIB']
    del job.attrib['INSTREAM_JCL']
    del job.attrib['USE_INSTREAM_JCL']
    return

def process_database(inner_tree, job, args):
    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdldatabase': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdldatabase',
        'jsdldatabase': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdldatabase'
    }        
    try:
        driverClass = inner_tree.find('.//jsdldatabase:driverClass', namespaces).text
        if (driverClass.startswith("oracle")):
            convert_oracle(inner_tree, job, args)
        else:
            print ("Unsupported DB Type - job", job.attrib['JOBNAME'])
            return
    except AttributeError:
        try:
            dbms = inner_tree.find('.//jsdldatabase:dbms', namespaces).text
            if (dbms.startswith("mssql")):
                convert_mssql(inner_tree, job, args)
            else:
                print ("Unsupported DB Type - job", job.attrib['JOBNAME'])
                logging.debug("Unsupported DB Type - job", job.attrib['JOBNAME'])
                return
        except AttributeError:
            print ("Unsupported DB Type - job", job.attrib['JOBNAME'])
            logging.debug( "Unsupported DB Type - job", job.attrib['JOBNAME'])
            for elem in inner_tree.iter():
                print(f"Tag: {elem.tag}, Attributes: {elem.attrib}, Text: {elem.text}")
            return

def convert_oracle(inner_tree, job, args):
    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdldatabase': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdldatabase'
    }       
    stored_procedure = inner_tree.find('.//jsdldatabase:storedProcedure', namespaces)
    stored_procedure_name = extract_variable(stored_procedure.attrib['name'])    
    if stored_procedure_name.strip == '':
        print("No stored procedure name found in job", job.attrib['JOBNAME'])
        return 
    job.attrib['APPL_TYPE'] = "DATABASE"
    job.attrib['APPL_FORM'] = "Databases"
    job.attrib['APPL_VER']  = "ANY"
    job.attrib['CM_VER']    = 'N/A'            
    db_url = extract_variable(inner_tree.find('.//jsdldatabase:connectionUrl', namespaces).text) 
    db_server = db_url.split(':')[0]
    db_port = db_url.split(':')[1]
    db_user = extract_variable(inner_tree.find('.//jsdl:userName', namespaces).text)
    add_var(job,"%%DB-ACCOUNT",db_server + ":" +  db_port + "_" + db_user)
    add_var(job,"%%DB-DB-TYPE","Oracle")    
    add_var(job,"%%DB-AUTOCOMMIT","N")
    add_var(job,"%%DB-DB_VERSION","Any")
    add_var(job,"%%DB-OUTPUT_FORMAT","Text")
    add_var(job,"%%DB-CSV_SEPERATOR",",")
    #add_var("%%DB-SSIS_PACKAGE_SOURCE","")  # <------l
    add_var(job,"%%DB-APPEND_LOG","Y") # <------
    add_var(job,"%%DB-STP_PACKAGE","*")
    add_var(job,"%%DB-EXEC_TYPE","Stored Procedure") 
    add_var(job,"%%DB-STP_NAME",stored_procedure_name)
    add_var(job,"%%DB-APPEND_OUTPUT","N")

    del job.attrib['MEMNAME']
    del job.attrib['MEMLIB']
    del job.attrib['INSTREAM_JCL']
    del job.attrib['USE_INSTREAM_JCL']
    return

def convert_mssql(inner_tree, job, args):
    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdldatabase': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdldatabase'
    }       
    job.attrib['APPL_TYPE'] = "DATABASE"
    job.attrib['APPL_FORM'] = "Databases"
    job.attrib['APPL_VER']  = "ANY"
    job.attrib['CM_VER']    = 'N/A'            

    db_server = extract_variable(inner_tree.find('.//jsdldatabase:server', namespaces).text)
    db_port = extract_variable(inner_tree.find('.//jsdldatabase:port', namespaces).text)
    db_user = extract_variable(inner_tree.find('.//jsdl:userName', namespaces).text)

    add_var(job,"%%DB-ACCOUNT",db_server + ":" +  db_port + "_" + db_user)
    add_var(job,"%%DB-DB-TYPE","MSSQL")    
    add_var(job,"%%DB-AUTOCOMMIT","N")
    add_var(job,"%%DB-OUTPUT_FORMAT","Text")
    add_var(job,"%%DB-CSV_SEPERATOR",",")
    add_var(job,"%%DB-SSIS_PACKAGE_SOURCE","SQL Server")
    add_var(job,"%%DB-APPEND_LOG","Y")
    add_var(job,"%%DB-STP_PACKAGE","*")
    try:
        stored_procedure = inner_tree.find('.//jsdldatabase:storedProcedure', namespaces)
        stored_procedure_name = extract_variable(stored_procedure.attrib['name'])    
        if stored_procedure_name.strip == '':
            print("No stored procedure name found in job", job.attrib['JOBNAME'])
            logging.debug("No stored procedure name found - job", job.attrib['JOBNAME'])
            return 
        add_var(job,"%%DB-STP_NAME",stored_procedure_name)
        add_var(job,"%%DB-EXEC_TYPE","Stored Procedure") 
        if stored_procedure_name.find('.') == -1:
            schema = 'dbo' # default for MSSQL
        else:
            schema = stored_procedure_name.split('.')[0]         
            add_var(job,"%%DB-STP_SCHEM",schema)    
    except AttributeError:
        try:
            sql = extract_variable(inner_tree.find('.//jsdldatabase:dbStatement', namespaces).text)
            if sql.strip() == '':
                print("No SQL to execute found in job", job.attrib['JOBNAME'])
                logging.error("No SQL to execute found - job", job.attrib['JOBNAME'])
                return
            add_var(job,"%%DB-EXEC_TYPE","Open Query") 
            add_var(job,"%%DB-QTXT-N001-SUBQTXT",sql )
            add_var(job,"%%DB-QTXT-N001-SUBQLENGTH",str(len(sql)))
            instance = inner_tree.find('.//jsdldatabase:database/jsdldatabase:sqlActionInfo/jsdldatabase:database',  namespaces).text
            add_var(job,"%%MSSQL-INSTANCE",instance)
        except AttributeError:    
            print("No SQL to execute found in job", job.attrib['JOBNAME'])
            logging.error("No SQL to execute found - job", job.attrib['JOBNAME'])
            return

    
    add_var(job,"%%DB-APPEND_OUTPUT","N")
    
    #add_var(job,"%%DB-SSA_NAME","")

    del job.attrib['MEMNAME']
    del job.attrib['MEMLIB']
    del job.attrib['INSTREAM_JCL']
    del job.attrib['USE_INSTREAM_JCL']    
    return

def process_restful(inner_tree, job, args):
    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdlrestful': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdlrestful'
    }        
    method = extract_variable(inner_tree.find('.//jsdlrestful:method', namespaces).text)
    if method.strip().upper() != 'POST':
        print("Unsupported HTTP method", method)
        return
    job.attrib['APPL_TYPE'] = "RST062023"
    job.attrib['APPL_FORM'] = "RST062023"
    job.attrib['APPL_VER']  = "ANY"
    job.attrib['CM_VER']    = 'N/A'            
    add_var(job,"%%UCM-ACCOUNT","DUMMY")
    add_var(job,"%%UCM-APP_NAME","RST062023")
    add_var(job,"%%UCM-METHOD","POST")
    add_var(job,"%%UCM-BODY_REQ_TYPE","text")
    add_var(job,"%%UCM-CONN_TIMEOUT","50")
    add_var(job,"%%UCM-APPEND_REQUEST","checked")
    add_var(job,"%%UCM-APPEND_RESPONSE","checked")
    uri = extract_variable(inner_tree.find('.//jsdlrestful:URI', namespaces).text)
    match = re.match(r'^([^\/]+(?:\/\/[^\/]*)?)\/(.*)', uri)
    endpoint = ''
    path = ''
    if match:
        endpoint, path = match.groups()
        if not path.startswith('/'):
            path = '/' + path
    else:
        print("Invalid URI format in job", job.attrib['JOBNAME'], uri)
        return
    add_var(job,"%%UCM-ENDPOINT_URL",endpoint)
    body = extract_variable(inner_tree.find('.//jsdlrestful:InputTextBody', namespaces).text)
    add_var(job,"%%UCM-BODY_BODY_REQUEST",body)
    add_var(job,"%%UCM-URL_REQUEST_PATH",path)
    add_var(job,"%%UCM-HEADERS_KEY_001","Content-Type")
    content_type = extract_variable(inner_tree.find('.//jsdlrestful:contentType', namespaces).text)
    add_var(job,"%%UCM-HEADERS_VALUE_001",content_type)
    try:
        headerKey = inner_tree.find('.//jsdlrestful:HeadersValue', namespaces).attrib['key']
        headerVal = extract_variable(inner_tree.find('.//jsdlrestful:HeadersValue', namespaces).text)
        add_var(job,"%%UCM-HEADERS_KEY_002",headerKey)
        add_var(job,"%%UCM-HEADERS_VALUE_002",headerVal)
    except AttributeError:
        pass # no headers
    add_var(job,"%%UCM-WS_OUTPUT_PARAMETERS_HTTPCODE_001","*")	
    filename =  extract_variable(inner_tree.find('.//jsdlrestful:outputFileName', namespaces).text)
    add_var(job,"%%UCM-WS_OUTPUT_PARAMETERS_VARIABLE_001","File:"+filename)

    del job.attrib['MEMNAME']
    del job.attrib['MEMLIB']
    del job.attrib['INSTREAM_JCL']
    del job.attrib['USE_INSTREAM_JCL']
    return    

def process_executable(inner_tree, job, args):
    namespaces = {
        'jsdl': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdl',
        'jsdle': 'http://www.ibm.com/xmlns/prod/scheduling/1.0/jsdle'
    }        
    job.attrib['APPL_TYPE'] = "OS"
    try:
        executable = inner_tree.find('.//jsdle:executable', namespaces)
    except AttributeError:
        print("No executable object found - job", job.attrib['JOBNAME'])
        logging.error("No executable object found - job", job.attrib['JOBNAME'])
        return
    if not executable.attrib ['workingDirectory'] is None and  executable.attrib ['workingDirectory'].strip() != '':
        workdir = extract_variable(executable.attrib ['workingDirectory'])
        job.attrib['OVERRIDE_PATH'] = workdir
    try:
        command = executable.attrib ['path'] 
        job.attrib['TASKTYPE'] = "Command"
        job.attrib['CMDLINE'] = command
        del job.attrib['MEMNAME']
        del job.attrib['INSTREAM_JCL']
        del job.attrib['USE_INSTREAM_JCL']
    except KeyError:
        try:
            script = inner_tree.find('.//jsdle:script', namespaces)
            #job.attrib['CMDLINE'] = "CMD.EXE"
            job.attrib['MEMNAME'] = "script." + script.attrib['suffix']
            job.attrib['INSTREAM_JCL'] = script.text
        except AttributeError:
            print("No path found and no script object found - job", job.attrib['JOBNAME'])
            logging.error( "No path found and no script object found - job", job.attrib['JOBNAME'])
            return
    del job.attrib['MEMLIB']
    return

def process_jtype_job(job, args):
    try:
        if job.attrib['USE_INSTREAM_JCL'] != 'Y':
            print('Job', job.attrib['JOBNAME'], 'is not using instream JCL, cannot convert it')
            logging.info( "Job is not using instream JCL - job" + job.attrib['JOBNAME'] +'cannot convert it')
            return
        #INSTREAM_JCL="JOBTYPE(/file transfer)
        instreamJcl = job.attrib['INSTREAM_JCL']
        if ('//*%OPC' in instreamJcl):
            print('Job', job.attrib['JOBNAME'], 'is using %OPC directives, please convert it manually')
            logging.info( "Job is using %OPC directives - job" + job.attrib['JOBNAME'] +'please convert it manually')
            return
        instreamJcl = instreamJcl.replace(')\n', '\001\002')
        instreamJcl = instreamJcl.replace('\n', '')
        instreamJcl = instreamJcl.replace('\001\002', ')\n')
        #global dict_of_keywords
        dict_of_keywords = {}
        for line in instreamJcl.split('\n'):
            kw = line[:line.find('(')].strip().upper()
            dict_of_keywords[kw] = line[line.find('(')+1:-1].strip()
        jobtype = dict_of_keywords['JOBTYPE'].lower()
        match jobtype:
            case '/file transfer':
                process_jtype_ftp(dict_of_keywords, job, args)
            case _:
                print( "Unsupported job type - job" , job.attrib['JOBNAME'] , "-" + jobtype[2:])
                logging.info( "Unsupported job type - job " + job.attrib['JOBNAME'] + " - " + jobtype[2:])
    except Exception as e:
        print('Unexpected error processing job', job.attrib['JOBNAME'], 'in application', job.attrib['APPLICATION'], ':', e)
        ###logging.error(' Unexpected error processing job {} in application {}: {}'.format(job.attrib['JOBNAME'], job.attrib['APPLICATION'], e))                
    return

def process_jtype_ftp(keywords, job, args):
    
    job.attrib['APPL_TYPE'] = "FILE_TRANS"
    job.attrib['APPL_FORM'] = "AFT"
    job.attrib['APPL_VER']  = "ANY"
    job.attrib['CM_VER']    = 'N/A'
       
    remote_host = extract_jclvar(keywords.get('SERVER',''))
    remote_user =  '%%OWNER'
    ftp_direction = 'receive' if extract_jclvar(keywords.get('TRANSFERTYPE','DOWNLOAD')).lower()=='download' else 'send'
    src_file = extract_jclvar(keywords.get('LOCALFILE',''))
    dest_file = extract_jclvar(keywords.get('REMOTEFILE',''))
    file_exist = 'REPLACE'
    transfer_mode =  extract_jclvar(keywords.get('TRANSFERMODE','TEXT')).lower()
    
    add_var(job, '%%FTP-ACCOUNT', 'local+' + remote_host+"_"+remote_user)

    add_var(job, '%%FTP-CONNTYPE1','LOCAL')
    add_var(job, '%%FTP-LHOST','Local')

    add_var(job, '%%FTP-CONNTYPE2','FTP')
    add_var(job, '%%FTP-RHOST',remote_host)
    add_var(job, '%%FTP-RUSER',remote_user)
    add_var(job, '%%FTP-RPASSIVE','0')
    add_var(job, '%%FTP-CONT_EXE_NOTOK','0')
    add_var(job, '%%FTP-RPF','1')
    add_var(job, '%%FTP-USE_DEF_NUMRETRIES','0')
    add_var(job, '%%FTP-TRANSFER_NUM','1')
    add_var(job, '%%FTP-AUTOREFRESH','False')
    add_var(job, '%%FTP-Is','5')
    
    if transfer_mode.lower() == 'binary':
        add_var(job, '%%FTP-TYPE1','B')
    else:
        add_var(job, '%%FTP-TYPE1','I')
        
    if ftp_direction.lower() == 'receive':    
        add_var(job, '%%FTP-UPLOAD1','1') # 1 for download, 0 for upload
        add_var(job, '%%FTP-RPATH1',dest_file)
        add_var(job, '%%FTP-LPATH1',src_file)
    else:
        add_var(job, '%%FTP-UPLOAD1','0') # 1 for download, 0 for upload       
        add_var(job, '%%FTP-RPATH1',src_file)
        add_var(job, '%%FTP-LPATH1',dest_file)

    if file_exist.lower() == 'replace':
        add_var(job, '%%FTP-IF_EXIST1','0')
    else: 
        add_var(job, '%%FTP-IF_EXIST1','1')
    del job.attrib['MEMNAME']
    del job.attrib['MEMLIB']
    del job.attrib['INSTREAM_JCL']
    del job.attrib['USE_INSTREAM_JCL']
    return

def get_var_content(job, name):
    var_value = ''
    try:
        for var in job.iter('VARIABLE'):
            if var.attrib['NAME'] == name:
                var_value = var.attrib['VALUE']
                break
    except KeyError:
        var_value = ' '
        logging.debug('Job {} has variable {} with no value'.format(job.attrib['JOBNAME'], name))
        print('Job {} has variable {} with no value'.format(job.attrib['JOBNAME'], name))
    return var_value

def set_var_content(job, name, value):
    for var in job.iter('VARIABLE'):
        if var.attrib['NAME'] == name:
            var.attrib['VALUE'] = value
            break
    return

def del_var(job, var_name):
    for var in job.iter('VARIABLE'):
        if var.attrib['NAME'] == var_name:
            job.remove(var)
    return

def get_var_dict(job):
    dict_of_vars:dict = {}
    for var in job.iter('VARIABLE'):
        try:
            dict_of_vars[var.attrib['NAME']] = var.attrib['VALUE']
        except KeyError:
            dict_of_vars[var.attrib['NAME']] = ''

    return dict_of_vars


def extract_variable(value):
    def replacer(match):
        varname = match.group(1)
        replacement = var_dictionary.get(varname, varname)
        return f"%%{replacement}"

    # Replace all ${...} occurrences
    retval = re.sub(r'\$\{([^}]+)\}', replacer, value)
    return retval

def extract_jclvar(value):
    def replacer(match):
        varname = match.group(1)
        replacement = var_dictionary.get(varname, varname)
        return f"%%{replacement}"

    # Replace all ${...} occurrences
    retval = re.sub(r'&([A-Za-z0-9_]+)\.', replacer, value)
                       
    return retval.replace("\\\\", "\\")
                          


def generate_log_file(log):
    dirpath = os.path.join("log")
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "jsdl-converter.log")
    if log == 2 and os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.INFO,format="%(asctime)s %(levelname)s %(name)s: %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.ERROR,format="%(asctime)s %(levelname)s %(name)s: %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.DEBUG,format="%(asctime)s %(levelname)s %(name)s: %(message)s", datefmt="%Y-%m-%d %H:%M:%S")

if __name__ == "__main__":
    start = time.time()
    main()
    end = time.time()
    print("It took:", math.floor((end-start)/60), "minutes and", math.floor((end-start)%60), "seconds")