#!/usr/local/bin/perl
################################################
#     CRON to CONTROL-M Conversion Utility     
#
# NOTE: the following lines used by CONTROL-M/Enterprise Manager
#   to identify the extenstion this script manages
#
# file_ext: crontab
# file_title: Crontab
# file_description: Crontab (Unix cron format)
#
################################################
use integer;
use Getopt::Std;
use File::Basename;

use Win32;
Win32::MsgBox("Starting",0,"Cron to CONTROL-M Conversion 6.4.08") if ($Debug);

$OUTPUTFILE = 'defjob.input';
$OUTPUTFORMAT = "deftable"; # or "defjob"
$CRONCTMLOG = 'cron-ctm.log';
$DATACENTER = 'datacenter';
$NODEID = '';
$NEWDAYSTART = '00:00';
$NEWDAYEND = '00:00';
$TABLENAME = 'crontab';
$QUITEMODE = false;

@months=('JAN','FEB','MAR','APR','MAY','JUN','JUL','AUG','SEP','OCT','NOV','DEC');

$desc = "";

##############################################
#         MAIN PROCEDURE STARTS HERE         #
##############################################

&process_args;
if ( not $QUITEMODE ) {
  &ask_questions;
}
&main_routine;
&done;

exit 0;

##############################################
# END MAIN PROCUDURE. SUBROUTINES BELOW HERE #
##############################################


sub process_args {
  getopts("i:f:d:s:n:e:t:o:l:q", \%opts) or die("Invalid usage");

  $CRONFILE = $opts{i} if defined $opts{i};
  $OUTPUTFORMAT = $opts{f} if defined $opts{f};
  $OUTPUTFORMAT = "defjob" if ($OUTPUTFORMAT != "deftable"); # make sure we have a valid format
  $DATACENTER = $opts{d} if defined $opts{d};
  $NODEID = $opts{s} if defined $opts{s};
  $NEWDAYSTART = $opts{n} if defined $opts{n};
  $NEWDAYEND = $opts{e} if defined $opts{e};
  $TABLENAME = $opts{t} if defined $opts{t};
  $OUTPUTFILE = $opts{o} if defined $opts{o};
  $CRONCTMLOG = $opts{l} if defined $opts{l};
  $QUITEMODE = true if defined $opts{q}; # non-interactive
}


sub ask_questions {
  $cmdcount=1;

######## ansi sequences for fancy output
  $ansih="\e[1m"; $ansil="\e[m";
  print "\nDoes your terminal support $ansih"."ansi"."$ansil escape sequences?\n";
  print "If the line above contains strange characters, answer No to this question [y]: ";
  $yesno=<STDIN>; chomp $yesno;
  if( not( $yesno=~/^$/  )) {
    if( not( $yesno=~/^y/i )) { $ansih=''; $ansil=''; }
  }
  
  print "\n* cron-ctm.pl - conversion utility from cron to CONTROL-M *\n\n";
  print "This procedure will read the crontabs that are on this \n";
  print "system and convert them to a set of CONTROL-M jobs. \n";
  print "Please answer the following questions carefully, these \n";
  print "have a great influence on the quality of the generated \n";
  print "output.\n";
  print "Most questions present a default answer between two brackets [].\n";
  print "If the default is okay for you, just press enter.\n";
  print "\n";

######## check user
  $idoutput=`id`;
  ($uid)=( $idoutput=~m/^uid=(\d+)\(/ );
  if( $uid ne "0" ) {
    print $ansih."You are not running this utility as root!".$ansil." This may cause \n";
    print "problems. This utility must have access to the crontabs, \n";
    print "and will also try to read all cronjobs to determine if \n";
    print "they are shell scripts.\n";
    print $ansih."Do you want to continue anyway [n] :";
    $yesno=<STDIN>; chomp $yesno;
    if( not( $yesno=~/^y/i )) { exit 1;}
    print "$ansil\n";
  }
  
  ### check location of crontabs
  if( -d "/var/spool/cron/crontabs/" ) { $CRONTABDIR='/var/spool/cron/crontabs/'; }   # Solaris,HP-UX
  elsif( -d "/var/cron/tabs/" )         { $CRONTABDIR='/var/cron/tabs/'; }              # Linux
  elsif( -d "/var/spool/cron/" )         { $CRONTABDIR='/var/spool/cron/'; }            # Linux
  else { $CRONTABDIR='' }
  print $ansih."In which directory on your system are your crontabs located?\n".$ansil;
  print "Enter a full pathname [$ansih$CRONTABDIR$ansil]: $ansih";
  $cronloc=<STDIN>; chomp $cronloc;
  if( $cronloc=~/^$/ ) { $cronloc=$CRONTABDIR; }
  while( not ( -d $cronloc ) ) {
    $save=$cronloc;
    print "$ansil\n$cronloc is not a directory.\nPlease reenter [$ansih$cronloc$ansil]: $ansih";
    $cronloc=<STDIN>; chomp $cronloc;
    if( $cronloc=~/^$/ ) { $cronloc=$save; }
  }
  if( not ( $cronloc=~m!/$! ) ) { $cronloc=$cronloc.'/'; }
  $CRONTABDIR = $cronloc;
  print "$ansil\n";

######## check datacenter
  $DATACENTER = `hostname`;
  chomp $DATACENTER;
  print $ansih."What is the datacenter name of your Control-M\/Server?\n".$ansil;
  print "You should enter the datacenter name as defined in your\n";
  print "datacenter definitions in Control-M\/Enterprise manager.\n";
  print "If you are not sure, just enter the hostname of the machine\n";
  print "that runs Control-M\/Server [$ansih$DATACENTER$ansil]: $ansih";
  $datac=<STDIN>; chomp $datac;
  if( $datac=~/^$/ ) { $datac=$DATACENTER; }
  while( $datac=~/\s/ ) {
    print "$ansil\nInput may not contain spaces.\nPlease reenter [$ansih$datac$ansil]: $ansih";
    $save=$datac;
    $datac=<STDIN>; chomp $datac;
    if( $datac=~/^$/ ) { $datac=$save; }
  }
  $DATACENTER=$datac;
  print "$ansil\n";
  
######## check nodeid
  $NODEID = `hostname`; # this default should usually be alright
  chomp $NODEID;
  print $ansih."What is the NODEID of your Control-M\/Agent?\n".$ansil;
  print "This should usually be the hostname of the machine you\n";
  print "are now running this utility on [$ansih$NODEID$ansil]: $ansih";
  $nodeid=<STDIN>; chomp $nodeid;
  if( $nodeid=~/^$/ ) { $nodeid=$NODEID; }
  while( $nodeid=~/\s/ ) {
    print "$ansil\nInput may not contain spaces.\nPlease reenter [$ansih$nodeid$ansil]: $ansih";
    $save=$nodeid;
    $nodeid=<STDIN>; chomp $nodeid;
    if( $nodeid=~/^$/ ) { $nodeid=$save; }
  }
  $NODEID=$nodeid;
  print "$ansil\n";

######## check newday start- and end-time
  $NEWDAYSTART= '00:00';  # START of newday procedure
  print "$ansih - NEWDAY time - $ansil\n";
  print "The following two parameters concern the time of the Control-M newday\n";
  print "Make sure you enter these right. All time calculations are based on\n";
  print "these parameters, entering wrong values here will cause jobs to be\n";
  print "scheduled at a wrong time.\n";
  print "This parameter can be found in your Control-M/Server, using \n";
  print "ctm_menu -> System Parameters -> Day Time \n\n";
  print "Enter start of newday (hh:mm) [$ansih$NEWDAYSTART$ansil]: $ansih";
  $newd=<STDIN>; chomp $newd;
  if( $newd=~/^$/ ) { $newd=$NEWDAYSTART; }
  while( not( ($h,$m) = ($newd=~/^(\d{1,2}):(\d\d)$/ )) ) {
    print $ansil."Please use hh:mm format [$ansih$newd$ansil]: $ansih";
    $save=$newd;
    $newd=<STDIN>; chomp $newd;
    if( $newd=~/^$/ ) { $newd=$save; }
  }
  $NEWDAYSTART=$newd;
  $m=($m+10);
  if( $m>59 ) { $m%=60; $h+=1; }
  if( $h>23 ) { $h%=24; }
  $NEWDAYEND=sprintf("%s:%02d",$h,$m);
  print $ansil."Enter estimated end of newday (hh:mm) [$ansih$NEWDAYEND$ansil]: $ansih";
  $newe=<STDIN>; chomp $newe;
  if( $newe=~/^$/ ) { $newe=$NEWDAYEND; }
  while( not( $newe=~/^(\d{1,2}):(\d\d)$/ ) ) {
    print $ansil."Please use hh:mm format [$ansih$newe$ansil]: $ansih";
    $save=$newe;
    $newe=<STDIN>; chomp $newe;
    if( $newe=~/^$/ ) { $newe=$save; }
  }
  $NEWDAYEND=$newe;
  print "$ansil\n";
  
######## check datacenter
  $TABLENAME='cron-'.$DATACENTER;
  print $ansih."Control-M Table name\n".$ansil;
  print "This will be the name of the scheduling table in which the\n";
  print "created job definitions are stored. I highly recommend to \n";
  print "choose a new (non-existing) name for this. This will make\n";
  print "it easier to delete all generated jobs from the Control-M/EM\n";
  print "database in the event that you need to rerun this utility.\n";
  print "Scheduling Table name [$ansih$TABLENAME$ansil]: $ansih";
  $schedt=<STDIN>; chomp $schedt;
  if( $schedt=~/^$/ ) { $schedt=$TABLENAME; }
  while( $schedt=~/\s/ ) {
    print "$ansil\nInput may not contain spaces.\nPlease reenter [$ansih$schedt$ansil]: $ansih";
    $save=$schedt;
    $schedt=<STDIN>; chomp $schedt;
    if( $schedt=~/^$/ ) { $schedt=$save; }
  }
  $TABLENAME=$schedt;
  print "$ansil\n";

  print "$ansil\nworking...\n";
}


sub done {
  print "...done!\n\n";
  print "Two files have been created:\n\n";
  print $ansih.$OUTPUTFILE.":\n".$ansil;
  print "This file contains the input for the Control-M/Enterprise Manager\n";
  print "utility deftable/defjob. Please read the Control-M/EM Utility Guide to find\n";
  print "out how to import the new jobs into the database\n\n";
  print $ansih.$CRONCTMLOG.":\n".$ansil;
  print "This file contains log messages created during the conversion of\n";
  print "your crontabs. It may contain useful information when you are trying\n";
  print "to figure out why certain jobs were created the way they were.\n\n";
  print $ansih."Don't forget to delete the jobs from the crontabs which are\n";
  print "now going to scheduled by Control-M!".$ansil;
  print "\n\n";
}


sub main_routine {
  if (defined $CRONFILE) {
    $CRONTABDIR = "";
    @cronfiles = $CRONFILE;
  } else {
    opendir (CRONDIR, $CRONTABDIR) || die "Can't open directory $CRONTABDIR: $!";
    @cronfiles = readdir(CRONDIR);
    closedir CRONDIR;
  }

  open (CONVLOG, ">$CRONCTMLOG");
  print CONVLOG "\ncron-ctm.pl v6.4.07 started\n";

  &{"init_xml_".$OUTPUTFORMAT};

  foreach $cronfile (@cronfiles) {
    next if ($cronfile eq '..') or ($cronfile eq '.'); #ignore . and .. dir entries
    if (defined $CRONFILE) {
      $xml_owner = basename($CRONFILE, ".crontab");
    } else {
      $xml_owner = $cronfile;
    }
    $file = $CRONTABDIR.$cronfile;
    $lineno = 0;
    open (CRON, $file) || die "Can't open file $file: $!";
    print CONVLOG "\nConverting crontab for $file ($OUTPUTFORMAT)\n";
    while (<CRON>) {
      $lineno = $lineno + 1;
       # ignore comments and empty lines 
      chomp;
      if (/^\s*(#|$)/)
      {
      	$desc = $desc.$_;
      	next;
      }
      	
      $cronline=$_;
      $cronline =~ s/^\s*//;
      ($minutes, $hours, $dom, $moy, $dow, $xml_cmd) = split(/\s+/,$cronline,6);
      print CONVLOG "  input: $cronline\n";
      &eval_dates;
      &determine_times;
      &determine_tasktype;
      &parse_command_line;
      &parse_desc;
      &write_xml;
    }
  }
  &{"close_xml_".$OUTPUTFORMAT};
  print CONVLOG "\ncron-ctm.pl ended\n\n";
  close (CONVLOG);
}


sub parse_command_line {
   $doubleq='&quot;';
   $singleq='&apos;';
   $lessth ='&lt;';
   $greatth='&gt;';
   $amp    ='&amp;';
   $xml_cmd =~ s/\&/$amp/g;
   $xml_cmd =~ s/\"/$doubleq/g;
   $xml_cmd =~ s/\'/$singleq/g;
   $xml_cmd =~ s/</$lessth/g;
   $xml_cmd =~ s/>/$greatth/g;
}
sub parse_desc {
   $doubleq='&quot;';
   $singleq='&apos;';
   $lessth ='&lt;';
   $greatth='&gt;';
   $amp    ='&amp;';
   $desc =~ s/\&/$amp/g;
   $desc =~ s/\"/$doubleq/g;
   $desc =~ s/\'/$singleq/g;
   $desc =~ s/</$lessth/g;
   $desc =~ s/>/$greatth/g;
}


sub eval_dates {
  # Month of year
  @MON=(0,0,0,0,0,0,0,0,0,0,0,0);
  if ($moy eq "*/2")
  {
    @MOY = (0,2,4,6,8,10);
  }
  elsif ($moy eq "*/3")
  {
    @MOY = (0,3,6,9);
  }
  elsif ($moy eq "*/4")
  {
    @MOY = (0,4,8);
  }
  elsif ($moy eq "*/6")
  {
    @MOY = (0,6);
  }
  elsif ($moy =~ /^\*$/) {
    @MOY=(0..11);
  } else {
    $moy =~ s/-/\.\./g;
    @MOY = eval("($moy)");
  }
  foreach $mon (@MOY) {
    $MON[$mon-1]=1;
  }
  $xml_months='';
  for( $i=0; $i<12; $i++) {
    $xml_months.="$months[$i]=\"$MON[$i]\"  ";
  }
  $PPP = join(" ", @MOY);
  print CONVLOG "    MONTHS  : $PPP\n";

  # weekdays and monthdays
  if(($dow =~ /^\*$/) && ($dom =~ /^\*$/)) {
    $xml_days    ='ALL';
    $xml_days_and_or='OR';
    $xml_weekdays='';
    print CONVLOG "    DAYS    : $xml_days\n";
  } 
  elsif ($dow =~ /^\*$/) {
    $dom =~ s/-/\.\./g;
    $xml_days = join(",", eval("($dom)"));
    $xml_days_and_or='OR';
    $xml_weekdays='';
    print CONVLOG "    DAYS    : $xml_days\n";
  }
  elsif ($dom =~ /^\*$/) {
    $xml_days    ='ALL';
    $dow =~ s/-/\.\./g;
    $xml_weekdays = join(",", eval("($dow)"));
    $xml_days_and_or='AND';
    print CONVLOG "    WEEKDAYS: $xml_weekdays\n";
  }
  else {
    $dom =~ s/-/\.\./g;
    $xml_days = join(",", eval("($dom)"));
    $dow =~ s/-/\.\./g;
    $xml_weekdays = join(",", eval("($dow)"));
    $xml_days_and_or='AND';
    print CONVLOG "    WEEKDAYS: $xml_weekdays\n";
    print CONVLOG "    DAYS    : $xml_days\n";
  }
}


sub determine_times {
## a bit if complicated math in this one. Don't change anything here ##
## unless you're absolutely sure you know what you are doing!        ##
  $xml_cyclic='0';
  $xml_interval=0;
  @xml_timefrom=();
  $xml_timeto='';
  $first_before_newday = 0;
  $xml_times_sequence = "";

  if( ($hours =~ /^\d+$/) && ($minutes =~ /^\d+$/) ) {
    $xml_cyclic='0';
    $xml_interval=0;
    @xml_timefrom=(@xml_timefrom,sprintf("%02d%02d",$hours,$minutes));
    printf CONVLOG "    TIMEFROM: %02d%02d\n",$hours,$minutes;
    ($h,$m)=($NEWDAYSTART=~/(\d+)\:(\d+)/); $newdaystartmins=$h*60+$m;
    ($h,$m)=($NEWDAYEND=~/(\d+)\:(\d+)/);   $newdayendmins=$h*60+$m;
    $jobmins=$hours*60+$minutes;
    if( ($jobmins > $newdaystartmins) && ($jobmins < $newdayendmins) ) {
      printf CONVLOG "    WARNING: this job is scheduled to run during Newday, thus may NEVER run!\n";
    }
  }
  
  elsif( ($hours =~ /^\*$/) && ($minutes =~ /^\*$/) ) {
    printf CONVLOG "    WARNING: asterisk (*) in the minute field.\n";
    $xml_cyclic='1';
    $xml_interval=1;
    ($newdayh,$newdaym)=($NEWDAYEND=~/(\d+)\:(\d+)/);
    @xml_timefrom=(@xml_timefrom,sprintf("%02d%02d",$newdayh,$newdaym));
    print CONVLOG "    determined this to be a cyclic job with a 1 minute interval\n";
  }

  else {
    if( $minutes =~ /^\*$/ ) {
      printf CONVLOG "    WARNING: asterisk (*) in the minute field.\n";
    }
    $hours   =~ s/\*/0-23/;
	$hours=expand_all_ranges($hours);
    if ( $hours =~ /\-/)
    {
#         $hours   =~ s/-/\,/g;
         ($FromHours,$ToHours) = split(/-/,$hours);
         if ($FromHours < $ToHours)
         {
         	$hours = $FromHours;
         	for ($HourIndex = $FromHours; $HourIndex < $ToHours; $HourIndex++)
         	{
          	$NewHour = $HourIndex + 1;
          	$hours = $hours . "," . $NewHour;
        	}
        }
        else
        {			
         	$hours = $FromHours;
         	for ($HourIndex = $FromHours; $HourIndex < 23; $HourIndex++)
         	{
          	$NewHour = $HourIndex + 1;
          	$hours = $hours . "," . $NewHour;
        	}
         	for ($HourIndex = -1; $HourIndex < $ToHours; $HourIndex++)
         	{
          	$NewHour = $HourIndex + 1;
          	$hours = $hours . "," . $NewHour;
        	}
        }
         
    }
    $minutes =~ s/-/\.\./g;
#    $hours   =~ s/\*/0\.\.23/;
    $minutes =~ s/\*/0\.\.59/;
 #   @hours=eval("($hours)");
    @hours = split(/,/, $hours);
    @minutes=eval("($minutes)");
    
    @times=();
    foreach $hour (@hours) {
      foreach $minute (@minutes) {
        $minuteofday=$hour*60+$minute;
        print CONVLOG "    for ",$hour," and ",$minute," is ",,$minuteofday,"\n";
        @times=(@times,$minuteofday);
      }
    }
    @times = sort {$a <=> $b} @times;
#    @times = sort @times;

    $num_entries = $#times;
      print CONVLOG "    found ",$num_entries,"\n";
    $xml_interval=($times[$#times]-$times[0])/$#times;
    $xml_cycic_type = "Interval";
    $xml_ind_cyclic = "TARGET";
    $xml_cyclic='1';
    for($i=1; $i<=$#times; $i++) { if( ($times[$i]-$times[$i-1])!=$xml_interval ) { $xml_cyclic='0'; } }
    if( $xml_cyclic eq '1' ) {
      print CONVLOG "    determined this to be a cyclic job with a $xml_interval minute interval\n";

      ## determine timeto
      $endmin =$times[$#times]+$xml_interval;
      if( $endmin>=1440 ) { undef( $xml_timeto ); }
      else {
        $endmin=($endmin+$newdaystartmins-1) % 1440;
        $h=$endmin/60; $m=$endmin % 60;
        $xml_timeto=sprintf("%02d%02d",$h,$m);
        print CONVLOG "    TIMETO: $xml_timeto\n";
      }

      ## determine timefrom 
      ($newdayh,$newdaym)=($NEWDAYSTART=~/(\d+)\:(\d+)/);
      $newdaystartmins=$newdayh*60+$newdaym;
      for( $i=0; $i<=$#times; $i++) {
        $times[$i]-=$newdaystartmins;
        if( $times[$i]<0 ) 
        { 
        	$times[$i]+=1440;
        	$first_before_newday = 1;
        }
      }
      @times = sort {$a <=> $b} @times;
#      @times = sort @times;
      
      ## determine time from
      $startmin=($times[0]+$newdaystartmins) % 1440;
      if( $first_before_newday eq 1 )
      {
          print CONVLOG "First run before New-Day. All runs before New-Day will be ignored";
          Win32::MsgBox("First run before New-Day. All runs before New-Day will be ignored");
      }

      if( $startmin lt $newdaystartmins ) { $startmin=($times[1]+$newdaystartmins) % 1440; }
      $h=$startmin/60; $m=$startmin % 60;
      @xml_timefrom=(sprintf("%02d%02d",$h,$m));
      print CONVLOG "    TIMEFROM: $xml_timefrom[0]\n";
	  
      ## determine time to
    } else {
      $xml_interval=0;
      $xml_cycic_type = "SpecificTimes";
      $xml_cyclic='1';
#      printf CONVLOG "    WARNING: Could not determine cyclic interval. These will become %d separate jobs.\n", $#times+1;
      foreach $hour (@hours) {
        foreach $minute (@minutes) {
#          @xml_times_sequence=(@xml_times_sequence,sprintf("%02d%02d",$hour,$minute));
          $times_sequence = sprintf("%02d%02d",$hour,$minute);
          $xml_times_sequence=$xml_times_sequence.$times_sequence.",";
          printf CONVLOG "    times_sequence: %02d%02d\n",$hour,$minute;
        }
      }
    }
  }
}


sub determine_tasktype {
  ### test if command on cmdline is shell script
  ($ln) = ($xml_cmd=~/^(.+)\s*$/);
  if( -e $ln ) {
    open (FILEH,$ln);
    read FILEH,$line,40; ### read first 60 bytes
    close (FILEH);
    if( $line=~/^(\#\!.+\/[ck]{0,1}sh)/ ) {
      $xml_tasktype="Job";
      printf CONVLOG "    TASKTYPE: Job\n";
      printf CONVLOG "      shell found: $1\n";
      $pos=rindex($ln,'/');
      $xml_memlib=substr($ln,0,$pos);
      $xml_memname=substr($ln,$pos+1);
      $xml_jobname=$xml_owner.' job '.$xml_memname;
    } else {
      printf CONVLOG "    TASKTYPE: Command\n";
      $xml_tasktype="Command";
      $pos=rindex($ln,'/');
      $xml_memname="command".$cmdcount;
      $xml_jobname=$xml_owner." command";
      $cmdcount++;
    }
  } else {
    printf CONVLOG "    TASKTYPE: Command\n";
    $xml_tasktype="Command";
    $xml_memname="command".$cmdcount;
    $pos=rindex($ln,'/');
    $xml_jobname=$xml_owner." command";
    $cmdcount++;
  }
}


##############################################
#         DEFJOB FORMAT
##############################################

sub init_xml_defjob {
  open (OUT, ">$OUTPUTFILE");
  print OUT '<?xml version="1.0" encoding="UTF-8"?>'."\n";
  print OUT '<!DOCTYPE DEFJOB SYSTEM "defjob.dtd">'."\n";
  print OUT '<DEFJOB>'."\n\n";
}

sub write_xml {

    print OUT '  <JOB'."\n";
    if ($OUTPUTFORMAT eq "defjob")
    {
    	print OUT '     DATACENTER="'.$DATACENTER."\"\n";
    	print OUT '     TABLE_NAME="'.$TABLENAME."\"\n";
    	print OUT '     TABLE_USERDAILY="SYSTEM"'."\n";
    }

    print OUT '     JOBNAME="'.$xml_jobname."\" \n";
    if( $xml_tasktype eq 'Command' ) {
      print OUT '     TASKTYPE="'.$xml_tasktype."\"\n";
      print OUT '     MEMNAME="'.$xml_memname."\"\n";
      print OUT '     CMDLINE="'.$xml_cmd."\"\n";
    } else {
      print OUT '     TASKTYPE="'.$xml_tasktype."\"\n";
      print OUT '     MEMNAME="'.$xml_memname."\"\n";
      print OUT '     MEMLIB="'.$xml_memlib."\"\n";
    }
    print OUT '     GROUP="crontab-'.$xml_owner."\" \n";
    print OUT '     APPLICATION="'.$TABLENAME."\" \n";
    print OUT '     NODEID="'.$NODEID."\"\n";
    print OUT '     OWNER="'.$xml_owner."\"\n";
    print OUT '     TIMEFROM="'.$xml_timefrom[0]."\"\n";
    if( $xml_timeto ) {
      print OUT '     TIMETO="'.$xml_timeto."\"\n";
    }
    if( $xml_cyclic eq '1' ) {
      print OUT '     CYCLIC="'.$xml_cyclic."\"\n";
      if ($xml_cycic_type eq "Interval")
      {
      	print OUT '     INTERVAL="'.sprintf("%05dM", $xml_interval)."\"\n";
      	print OUT '     IND_CYCLIC="TARGET" '."\n";
      	print OUT '     CYCLIC_TYPE="Interval"'."\n";
      }
      if ($xml_cycic_type eq "SpecificTimes")
      {
      	print OUT '     CYCLIC_TIMES_SEQUENCE="'.$xml_times_sequence."\"\n";
      	print OUT '     IND_CYCLIC="START" '."\n";
      	print OUT '     CYCLIC_TYPE="SpecificTimes"'."\n";
      	print OUT '     CYCLIC_TOLERANCE="10"'."\n";
      }

    }
    print OUT '     '.$xml_months."\n";
    print OUT '     DAYS="'.$xml_days."\"\n";
    print OUT '     DAYS_AND_OR="'.$xml_days_and_or."\"\n";
    if( $xml_weekdays gt '' ) {
      print OUT '     WEEKDAYS="'.$xml_weekdays."\"\n";
    }
    print OUT '     AUTHOR="cron-ctm.pl" '."\n";
    print OUT "     DESCRIPTION=\"",$desc," crontab line ",$lineno,"\"\n";
    $desc = "";
    print OUT '    >'."\n";
    print OUT '  </JOB>'."\n\n";

}

sub close_xml_defjob {
  print OUT '</DEFJOB>'."\n";
  close (OUT);
}

##############################################
#         DEFTABLE FORMAT
##############################################

sub init_xml_deftable {
  open (OUT, ">$OUTPUTFILE");
  print OUT "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
  print OUT "<!DOCTYPE DEFTABLE SYSTEM \"deftable.dtd\">\n";
  print OUT "<DEFTABLE>\n";
  print OUT " <SCHED_TABLE DATACENTER=\"$DATACENTER\" TABLE_NAME=\"$TABLENAME\" TABLE_USERDAILY=\"SYSTEM\">\n";
}


sub close_xml_deftable {
  print OUT " </SCHED_TABLE>\n";
  print OUT "</DEFTABLE>\n";
  close (OUT);
}

sub expand_all_ranges
{
	my $string = shift;
	
	while ($string=~/(.*?)(([0-9]+)-([0-9]+))(.*)/)
	{
		my $head=$1;
		my $tail=$5;
		my $to_replace =$2;
		my $range_start = $3;
		my $range_end=$4;		
		
		my $replacement_string=",";
		for (my $i=$range_start; $i<=$range_end; $i++)
		{
			$replacement_string.=$i;
			$replacement_string.=",";
		}
		$string=$head.$replacement_string.$tail;
	}
	$string=~s/,,/,/g;
	$string=~s/,$//;
	$string=~s/^,//;
	
	return $string;
}
