#!/usr/bin/perl
#use strict;
#use warnings;


#####################################################################################
# Authors: gshimi and ishifman - October 2015
# Version: 1
# Description: This script is meant to be invoked automatically by the CTM shout-to-program feature with the correct arguments and their values.
# Assumptions:
# 1. This script is executed on a machine with a CTM Server V8 or up installation.
# 2. A MU dependency file named MuDependencies.txt is located in the same directory as the script and contains the MU dependencies information in the format exported from Dollar Universe environment.
# Arguments:
# The nodes eligible to run the jobs are calculated from 3 of the scripts arguments:
# 1. -host_group - The logical name of the MU to relate to.
# 2. -relationship - One of (specific | none | parents | children | brothers), if not supplied defaults to children, representing the relationship to the value of -host_group's value.
# 3. -MU_type - A string containing one or two alphanumeric characters representing the type of MUs to filter by.
# 4. -job_to_order - The name of the CTM Job to be ordered by the ctmorder commands (one order on each eligible node).
# 5. -folder - The name of the CTM Folder containing the -job_to_order.
# 6. -condition_variable - If the Conversion Tool boolean rule for adding conditions is checked, add this string to condition names between jobs.
# 7. -debug - a boolean flag, if exists enables debugging.
#####################################################################################

my $LOGFILE_HANDLE = undef;
my $debug = 0;	# supply "-debug" in script args or set to '1' to enable log printing of debug entries
my $logFileDir = "";
my $logFilePath = "";

# args literal names and hash
my $host_group_arg_name = "-host_group";
my $MU_type_arg_name = "-MU_type";
my $relationship_arg_name = "-relationship";
my $job_to_order_arg_name = "-job_to_order";
my $job_parent_folder_arg_name = "-folder";
my $add_condition_arg_name = "-condition_variable";
my $folder_id_arg_name = "-folder_id";
my $debug_arg_name = "-debug";
my %argsMap = ();

my $agentName = shift(@ARGV);
my $args_string = shift(@ARGV);
my $urgancy = shift(@ARGV);



&open_log_file();
&init_args_map();
&printArgsToLog() if($debug == 1);

# from this point on - %argsMap is populated with mandatory values and default values
if(&validate_args())
{
	# from this point on - all args were validated - actual work may proceed
	my @nodes_to_run_on = &get_nodes_to_run_on();
	&perform_ctmorder(\@nodes_to_run_on);	
}
else
{
	# args validation failed - report and abort
	my $errorMessage =  "Given arguments validation failed!!\nAborting...\n";
	&ctmshout($errorMessage);
	&print_log($errorMessage);
}
&close_log_file();

##########
# subroutines
##########

sub open_log_file
{
	if($debug == 0)
	{
		return;
	}
	$logFileDir = "" if !defined $logFileDir;
	$logFileDir = $logFileDir."\\" if $logFileDir ne "";
	my $logFileName = &createLogFileName();
	$logFilePath = $logFileDir.$logFileName;
	
	verbose("open log file $logFilePath...");
	open $LOGFILE_HANDLE,">>".$logFilePath or ctmshout_and_die("failed to open log file at: $logFilePath");
}

sub createLogFileName
{
	my %args = &getArgsHashFromMessage();
	if(exists $args{$host_group_arg_name} && $args{$add_condition_arg_name} )
	{
		my $hostname = $args{$host_group_arg_name};
		my $orderid = 	$args{$add_condition_arg_name};
		
		return $hostname."_ORDERID_".$orderid."_PID_".$$.".log" 
	}
	else
	{
		return "order".$$.".log";
	}	
}

# This sub goes through the message part of the ctmshout and fills a map of key-value with the given arguments.
# It uses a regular expression to extract key-value pairs and/or flags (i.e. - only key, no values).
# Argument string example format: "-key1 value1 -flag1 -key2 "value 2" ... -keyN valueN"
# Non mandatory arguments are initialized with default values if not supplied.
# Arguments:
# agent name (not used by this script)
# message (actual script arguments)
# urgency (unused)
sub init_args_map
{
	my $arg_name = undef;
	my $arg_value = undef;
	
	if(not defined $args_string)
	{
		&ctmshout_and_die("no arguments supplied!!");
	}
	
	# capturing group #1 - leading whitespace, '-', word characters, stop at whitespace or end of string.
	# capturing group #2 - (quoted or not quoted) not '-' prefix, word characters (including '-' in it), stop at whitespace or end of string
	%argsMap = &getArgsHashFromMessage();
	# init default values if not provided
	if(not exists($argsMap{$MU_type_arg_name}) or
		not defined($argsMap{$MU_type_arg_name}) or
		($argsMap{$MU_type_arg_name} eq ""))
	{
		$argsMap{$MU_type_arg_name} = "*";	# default - ALL types
	}
	if(not exists($argsMap{$relationship_arg_name}) or
		not defined($argsMap{$relationship_arg_name}) or
		($argsMap{$relationship_arg_name} eq ""))
	{
		$argsMap{$relationship_arg_name} = "children";	# default - sons relation
	}	
	
}

sub getArgsHashFromMessage
{
	my %map ;
	while($args_string =~ /\s*((?<!\w)-[^\s]+?)(?:\s+|$)((?:[^-]".*?"|[^-][^\s]*))?\s*/ig)
	{
		# initialize hash with args keys and values
			my $arg_name = $1;
			my $arg_value = $2;
		if(defined $arg_name and not defined $arg_value)
		{
			if($arg_name eq $debug_arg_name)
			{
				$debug = 1;
			}
				$map{$arg_name} = "";
			next;
		}
		else
		{
				$map{$arg_name} = $arg_value if(defined $arg_name and defined $arg_value);
		}
	}
	return %map;
	}

sub printArgsToLog
	{
		&print_log("script was given below arguments and values:\n");
		foreach my $key (keys(%argsMap))
		{
			&print_log("$key => $argsMap{$key}\n");
		}
	}

# Validating the arguments map post its initialization.
# Verifying mandatory arguments exist and contain values
sub validate_args
{
	my $result = 0;	# false
	my $errorMessage;

	if(not exists($argsMap{$host_group_arg_name}) or
		not exists($argsMap{$job_to_order_arg_name}) or
		not exists($argsMap{$job_parent_folder_arg_name}))
	{
		# error handling - missing mandatory argument(s) - report and abort
		$errorMessage = "missing mandatory argument(s) - ($host_group_arg_name | $job_to_order_arg_name | $job_parent_folder_arg_name)";
		&ctmshout($errorMessage);
		&print_log($errorMessage);
	}
	else
	{
		if(not &defined_or_empty($argsMap{$host_group_arg_name}) or
			not &defined_or_empty($argsMap{$job_to_order_arg_name}) or
			not &defined_or_empty($argsMap{$job_parent_folder_arg_name}))
		{
			# error handling - missing mandatory argument(s) value - report and abort
			$errorMessage = "mandatory argument(s) missing value(s) - ($host_group_arg_name | $job_to_order_arg_name | $job_parent_folder_arg_name)";
			&ctmshout($errorMessage);
			&print_log($errorMessage);
		}
		else
		{
			$result = 1;
		}
	}
	if(length($argsMap{$MU_type_arg_name}) > 2
		or length($argsMap{$MU_type_arg_name}) == 1 && $argsMap{$relationship_arg_name} eq "brothers")
	{
		# error handling - missing mandatory argument(s) value - report and abort
		$errorMessage = "$MU_type_arg_name length must not exceed 2 characters, relation of \"brothers\" MUST contain exactly 2 characters.\ngiven value - ($argsMap{$MU_type_arg_name})";
		&ctmshout($errorMessage);
		&print_log($errorMessage);
		$result = 0;
	}
	return $result;
}

# Initializing the nodes to run on list according to the given relationship ("none" or "specific" sets the same (single) value to the list).
sub get_nodes_to_run_on
{
	my $relation = $argsMap{$relationship_arg_name};
	my $host_group = $argsMap{$host_group_arg_name};
	my @nodes_to_run_on = ();
	
	
	if (lc($relation) eq "brothers" or
		lc($relation) eq "children" or
		lc($relation) eq "parents")
	{
		@nodes_to_run_on = &filter_mu_tree();
	}
	else
	{
		# NONE/SPECIFIC MU: keep the same mu
		if(lc($relation) eq "none" or
			lc($relation) eq "specific")
		{
			push(@nodes_to_run_on, $host_group);
		}
		else
		{
			&ctmshout_and_die("$relationship_arg_name given value of $relation is not supported");
		}		
	}
	return @nodes_to_run_on;
}

# Iterating over the eligible nodes to run on list, composing the ctmorder command and executing it dynamically on the fly.
# Notifies if any of the nodes does not exist or any of the ctmorder executions fails.
sub perform_ctmorder
{
	my @nodes_to_run_on = @{shift()};
	my $condition = $argsMap{$add_condition_arg_name};
	my $folder = $argsMap{$job_parent_folder_arg_name};
	my $job = $argsMap{$job_to_order_arg_name};
	my $folderId = $argsMap{$folder_id_arg_name};
	my $ctmorder_to_run = "ctmorder -FOLDER $folder -JOBNAME $job -FORCE Y -ODATE ODAT -VARIABLE %%NODEGRP ";
	
	foreach my $node_to_run_on (@nodes_to_run_on)
	{
		&verbose("check NODEGRP value : $node_to_run_on")if(defined $node_to_run_on);
		my $isExistAgent = 1;
		my $isExistHostGroup = 1;
		my $ctm_order_command = undef;
		
		if(!defined $node_to_run_on || $node_to_run_on eq "" || $isExistAgent ||  $isExistHostGroup)
		{
			$ctm_order_command = $ctmorder_to_run.$node_to_run_on;
			$ctm_order_command .= " -INTO_FOLDER_ORDERID $folderId";
			if(defined($condition))
			{
				$ctm_order_command .= " -VARIABLE %%Monitoring_View_Variable $condition";
			}
			sleep(1);
			&send_ctm_order_command($ctm_order_command);
		}
		else
		{
			my $errorMessage = "non valid ctmorder request: ".$node_to_run_on." does not exist as a host group or host.\nThe command will not be executed";
			&ctmshout($errorMessage);
			&print_log($errorMessage);
		}		
	}
}

sub defined_or_empty
{
	my $string_to_check = shift;
	return (defined($string_to_check) && $string_to_check ne "")
}

# Calculating the full path for the mu dependencies file.
sub get_mu_deps_filename
{
	my $dir_seperator = "\\\\";
	if($^O ne "MSWin32")
	{
		$dir_seperator = "/";
	}
	my @path_parts = split($dir_seperator, __FILE__);
	pop @path_parts;
	my $mu_dep_pathname = join($dir_seperator, @path_parts);
	my $mu_dep_filename;
	if($mu_dep_filename eq "")
	{
		$mu_dep_filename = "MuDependencies.txt";
	}
	else
	{
		$mu_dep_filename = $mu_dep_pathname."\\MuDependencies.txt";
	}
	return $mu_dep_filename;
}

# Actual handling of the dependencies file.
sub filter_mu_tree
{
	my @nodes_list = ();
	my $MU_DEP_HANDLE = undef;
	my $mu_dep_filename = &get_mu_deps_filename();
	open $MU_DEP_HANDLE, "<", $mu_dep_filename or ctmshout_and_die("failed to open $mu_dep_filename - aborting\n");
	my @dep_mu_entries = &remove_headers(<$MU_DEP_HANDLE>);
	if(&mu_found_in_file(\@dep_mu_entries))
	{
		@nodes_list = &get_nodes_by_relation_and_type(@dep_mu_entries);	
	}
	else
	{	
		my $errorMessage = "requested MU ($argsMap{$host_group_arg_name}) was not found in the MU dependencies file.\n";
		&ctmshout($errorMessage);
		&print_log($errorMessage);
	}
	close $MU_DEP_HANDLE or ctmshout_and_die("failed to close mu_deps.txt");
	return @nodes_list;
}

sub mu_found_in_file
{
	my @dep_mu_entries = @{shift()};
	my $is_node_found_in_file = 0;
	my $node_to_relate_to = $argsMap{$host_group_arg_name};
	foreach my $entry (@dep_mu_entries)
	{
		if($entry =~ m/$node_to_relate_to/)
		{
			$is_node_found_in_file = 1;
		}
	}
	return $is_node_found_in_file;
}

# Removing the headers from the dependencies file.
sub remove_headers
{
	my @dep_mu_entries = @_;
	foreach my $line(@dep_mu_entries)
	{
		unless($line =~ m/^\\s*-+\\s+-+\\s*$/gi)
		{
			shift @dep_mu_entries;
		}
		shift @dep_mu_entries;
		last;
	}
	return @dep_mu_entries;
}

# Actual filtering the dependencies file according to host, relation and type provided.
sub get_nodes_by_relation_and_type
{
	my @nodes_list = ();
	my $relation = $argsMap{$relationship_arg_name};
	my $node_type = $argsMap{$MU_type_arg_name};
	my $node_to_relate_to = $argsMap{$host_group_arg_name};
	my @dep_mu_entries = @_;
	my ($mu, $dep_mu) = undef;	
		
	if(lc($relation) eq "children")
	{
		@nodes_list = &get_children_by_type(\@dep_mu_entries, $node_to_relate_to, $node_type);
	}
	elsif(lc($relation) eq "brothers")
	{
		my $parent_type = substr($node_type, length($node_type) - 1, 1);
		my $child_type = substr($node_type, 0, 1);
		my @parent_nodes = &get_parents_by_type(\@dep_mu_entries,$node_to_relate_to, $parent_type);
		foreach my $parent (@parent_nodes)
		{
			push(@nodes_list, &get_children_by_type(\@dep_mu_entries,$parent ,$child_type));
		}
		@nodes_list = deleteNode(\@nodes_list,$node_to_relate_to);
		
	}
	elsif(lc($relation) eq "parents")
	{
		@nodes_list = &get_parents_by_type(\@dep_mu_entries, $node_to_relate_to, $node_type);
	}
	if($debug == 1)
	{
		&print_log("nodes eligible to run on according to given arguments:\n");
		foreach my $node (@nodes_list)
		{
			&print_log("$node ") if(defined $node && $node ne "");
		}
	}
	return @nodes_list;
}

# Removing a node from the list
sub deleteNode
{
	my @nodes_list = @{shift()};
	my $nodeToRemove = shift();
	for ( my $index = 0  ; $index <= $#nodes_list; $index++ )
	{
		splice @nodes_list,$index, 1 if ($nodes_list[$index] eq $nodeToRemove);  # remove certain elements
 	}

	return @nodes_list;
}

sub get_children_by_type
{
	my @dep_mu_entries = @{shift()};
	my $node_to_relate_to = shift();
	my $type = shift();
	my @nodes_list = ();
	foreach my $entry (@dep_mu_entries)
	{
		my $is_correct_type = 0;
		chomp $entry;
		$entry = &trim($entry);
		my ($mu, $dep_mu) = ($entry =~ m/\s*([^\s]+)\s+([^\s]+)\s*/);
		$is_correct_type = 1 if($dep_mu =~ /^$type/i);
		push(@nodes_list, $dep_mu) if(($mu) eq $node_to_relate_to and $is_correct_type != 0)
										 									
	}
	return @nodes_list;
}

sub trim
{
	my $string = shift();
	$string =~ s/^\s+|\s+$//g;
	return $string; 
}

sub get_parents_by_type
{

	my @dep_mu_entries = @{shift()};
	my $node_to_relate_to = shift();
	my $type = shift();
	my @nodes_list = ();
	foreach my $entry (@dep_mu_entries)
	{
		my $is_correct_type = 0;
		chomp $entry;
		$entry = &trim($entry);
		my ($mu, $dep_mu) = ($entry =~ m/\s*([^\s]+)\s+([^\s]+)\s*/);
		$is_correct_type = 1 if($mu =~ /^$type/i);
		push(@nodes_list, $mu) if(($dep_mu) eq $node_to_relate_to and $is_correct_type != 0)
										 									
	}
	return @nodes_list;
}

# Executing the ctmorder command provided to it (taking OS type into consideration), reporting any errors if occured.
sub send_ctm_order_command
{
	# concat given nodegrp value to ctmorder command as the value of the -VARIABLE %%NODEGRP value
	my $command = shift;
	$command = $command." 2>&1";
	$command = $command.";" if($^O ne "MSWin32");#if its not windows OS add ';' for the 2>&1
	&print_log("run ordering command: ".$command);
	my $res = `$command`;
	$res =~ s/usage(.*\n.*)*//i;
	if($? ne 0)
	{		
		my $original_debug_value = $debug;
		
		# ALWAYS print failed ctmorder command to log	
		if($debug == 0)
		{
			$debug = 1;
			&open_log_file();
		}
		my $errorMessage = "CTMORDER FAILED - see details in log file - $logFilePath, located under same path as the script";
		&ctmshout($errorMessage);
		&print_log("CTMORDER FAILED -\nfailed command:\n\"$command\"\noutput:\n$res\n");
		if($original_debug_value == 0)
		{
			&close_log_file();
		}
		# reassign $debug original value
		$debug = $original_debug_value;
	}
	&print_stars_line_to_log(); 
	&print_log("output:\n $res");
	&print_stars_line_to_log();
}

# performing a ctmshout to EM with a runtime error message.
sub ctmshout
{
	
	my $errorMessage = shift();
	$errorMessage =~ s/"/\\"/g;
	my $scriptPath = __FILE__;
	$errorMessage = "Alerted from Shout Destination.\tProgram: ".$scriptPath."\tError: ".$errorMessage;
	$errorMessage = &escape_autoedits($errorMessage);
	if(length($errorMessage) > 256)
	{
		$errorMessage = substr( $errorMessage, 0, 255 );
	}
	$errorMessage = "\"".$errorMessage."\"";
	system("ctmshout -DEST EM -MESSAGE $errorMessage -SEVERITY u");
}

sub escape_autoedits
{
	my $errorMessage = shift();
	if(defined($errorMessage))
	{
		$errorMessage =~ s/%%/%%#/g;
	}
	return $errorMessage;
}



# Verifying given string appears or not in the given command's output.
sub isStringInCommandResult
{ 	
	my $string = shift();
	my $command_to_search_in = shift();
	my $res = `$command_to_search_in`;
	my @lines = split /^/m, $res;
	&verbose("lines: ");
	&verbose("@lines");
	my @linesWithGroupName = grep(/(^|.*\s+)$string\s+/i, @lines);
	&verbose("lines with group name: ");
	&verbose("@linesWithGroupName");
	my $isExist = @linesWithGroupName != 0; 
	&verbose("Result: true ") if($isExist);
	return $isExist ;
}

sub verbose
{
	my $string = shift(); 
	&print_log($string);
}

# Printing to log file only if debug was enabled.
sub print_log
{
	if($debug == 1)
	{
		my $string = shift();
		my $timestamp = localtime(time);
		print $LOGFILE_HANDLE $timestamp .": ". $string."\n" if(defined $LOGFILE_HANDLE);
	}
}

sub print_stars_line_to_log
{
	my $starsLine =  "***********************************\n";
	print $LOGFILE_HANDLE $starsLine if(defined $LOGFILE_HANDLE);
}

sub close_log_file
{
	if($debug == 0)
	{
		return;
	}
		close($LOGFILE_HANDLE) if(defined($LOGFILE_HANDLE)) or ctmshout_and_die("failed to close LOGFILE_HANDLE");
}

# Performing ctmshout with given error message and then dying in case of script terminal error encountered.
sub ctmshout_and_die
{
	my $message = shift();
	&print_log($message);
	(&ctmshout($message) && die "$message\n");
}
##########
# end subroutines
##########
