#!/usr/bin/python
#coding=utf8

from xml.dom import minidom
import xml.etree.cElementTree as ET			
import os, time, math		
import argparse
import copy
import logging
import datetime
from datetime import datetime

def argument_parser():
    parser = argparse.ArgumentParser(
       description="Convert MF jobs to DS")
    parser.add_argument('-i', dest='input_file', type=str, default='prodjob_fromEM.xml', help='Input XML exported from MF Control-M', metavar='')
    parser.add_argument('-o', dest='output_file', type=str, default='out.xml', help='Location for job definition xml where all job uses the DS job type', metavar='')
    parser.add_argument('-c', dest='compare_file', type=str, help='Input file in the output format', metavar='', default='')
    parser.add_argument('-d', dest='converted_dummy', type=str, help='Were mainframe jobs already converted to Dummy?', metavar='', default='yes')
    parser.add_argument('-r', dest='run_as_dummy', type=str, help='Transfrom DUMMY jobs to run-as-dummy jobs. Default: Yes', metavar='', default='Y')
    parser.add_argument('-t', dest='job_type', type=str, 
                        help='The target job type to be used in both APPL_TYPE, APPL_FORM and APP_NAME. Use MFL0242024 for Micro Focus AI/IF. Default: Command', metavar='', default='Command')
    parser.add_argument('-cmdline', dest='comamnd_line', type=str, 
                        help='The comamnd line to be used when the target job type is Command. %%MEMLIB in the command will have the MEMLIB field content', 
                        metavar='', default='run %%JOBNAME')
    parser.add_argument('-m', dest='controlm', type=str, help='The target DS Control-M name', metavar='', default='CTMDS')
    parser.add_argument('-n', dest='nodeid', type=str, help='The target DS HOST name. default: controlm name', metavar='', default='')
    parser.add_argument('-u', dest='unique', type=str, help='Should job names be forced to be unique in a folder - Y/N', metavar='', default='Y')
    parser.add_argument('-offset', dest='cond_days_offset', type=str, help='Should days offset in condition date kept as is - Y/N', metavar='', default='Y')
    parser.add_argument('-log', dest='log', type=int, help=argparse.SUPPRESS, default=1, metavar='')
    return parser

def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info(get_time_stamp() + ' script parameters: ' + str(args))

    print ('Input is: ' + args.input_file)
    print ('Output is: ' + args.output_file)
    if len(args.compare_file) > 0:
        print ('Compare file is: ' + args.compare_file)

    tree = ET.parse(args.input_file); print("Read the input file: " + args.input_file)
    root = tree.getroot()

    if len(args.compare_file) > 0:
        tree.write(args.compare_file);print("Original saved for comparison.")

    process_all_folders(root, args)

    print("Writing updates to the XML file...")

    tree.write(args.output_file);print("Updates saved to the output XML file.")

def process_all_folders(root, args):
    for folder in root.iter("FOLDER"):
        convert_folder(folder, args)
    for folder in root.iter("SMART_FOLDER"):
        convert_folder(folder, args)
    return

def convert_folder(folder, args):
    folder.attrib['PLATFORM'] = "UNIX"
    folder.attrib['TYPE'] = "1"
    folder.attrib["DATACENTER"] = args.controlm
    del folder.attrib["FOLDER_DSN"]
    warn_if_attribute_exists_and_remove("folder", folder.attrib["FOLDER_NAME"], folder, "TO_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("folder", folder.attrib["FOLDER_NAME"], folder, "FROM_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("folder", folder.attrib["FOLDER_NAME"], folder, "DUE_OUT_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("folder", folder.attrib["FOLDER_NAME"], folder, "PREV_DAY")
    convert_condition_date(folder, args)
    convert_condition_logic(folder)
    for child in folder:
        if child.tag.find("RULE_BASED_CALENDAR") == 0:
            warn_if_attribute_exists_and_remove("RBC", folder.attrib["FOLDER_NAME"]+'+'+child.attrib["NAME"], child, "PREV_DAY")

    process_all_jobs(folder, args)
    folder.attrib["FOLDER_NAME"] = folder.attrib["FOLDER_NAME"].replace("$","_")
    return

def warn_if_attribute_exists_and_remove(entity_type, entity_name, entity, attribute):
    try:
        attr_value = entity.attrib[attribute]
        if len(attr_value) > 0:
            logging.warning(get_time_stamp() + 'In {} {} unsupported attribute {} with value {} was removed'.format(entity_type, entity_name, attribute, attr_value))
            del entity.attrib[attribute]
    except KeyError:
        pass
    return

def process_all_jobs(folder, args):
    dict_of_jobnames:dict = {}
    for job in folder.iter('JOB'):
        logging.debug(get_time_stamp() + 'Processing job {} '.format(job.attrib['MEMNAME']))
        # if (job.attrib['TASKTYPE'] == 'Job') or (job.attrib['TASKTYPE'] == 'Dummy' and get_var_content(job,'%%$BMCWAIORIGTYPE') == 'Job'):
        try:
            if (job.attrib['TASKTYPE'] == 'Job' or job.attrib['TASKTYPE'] == 'Cyclic Job'):
                process_mf_job(job, args, dict_of_jobnames)
        except KeyError:
            pass
        else:
            if 'Y' in args.converted_dummy.upper():
                try:
                    if (job.attrib['TASKTYPE'] == 'Dummy'and get_var_content(job,'%%$BMCWAIORIGTYPE') == ''):
                        process_mf_job(job, args, dict_of_jobnames)
                except KeyError:
                    pass

                # print('job {} processed'.format(get_jobname(job)))
    return

def process_mf_job(job, args, dict_of_jobnames):
    logging.debug(get_time_stamp() + 'Converting job {} with type {}'.format(job.attrib['MEMNAME'], job.attrib['TASKTYPE']))

    convert_job_to_ds(job, args, dict_of_jobnames)
    
    memlib:str = job.attrib['MEMLIB']
    
    if (args.job_type == 'MFL0242024'):
        convert_job_to_MFL0242024(job, args)
    if (args.job_type == 'Command'):
        convert_job_to_comamnd(job, args)
    handle_dummy(job, memlib, args)
    
    return

def handle_dummy(job, memlib, args):
    overlib = get_overlib(job)
    if (memlib == "DUMMY" and overlib == "") or (overlib == "DUMMY"):
        tasktype: str = job.attrib['TASKTYPE']
        job.attrib['TASKTYPE'] = 'Dummy'
        if "Y" in args.run_as_dummy.upper():
            add_var(job, '%%$BMCWAIORIGTYPE', tasktype)
        clear_overlib(job)
    return

def convert_job_to_comamnd(job, args):
    job.attrib['APPL_TYPE'] = 'OS'
    job.attrib['TASKTYPE'] = 'Command'
    memlib = job.attrib['MEMLIB']
    cmdline = args.comamnd_line.replace("%%MEMLIB", memlib)
    job.attrib['CMDLINE'] = cmdline

    del job.attrib['MEMLIB']
    # del job.attrib["OWNER"]



    return

def convert_job_to_MFL0242024(job, args):
    
    job.attrib['APPL_TYPE'] = args.job_type
    job.attrib['APPL_FORM'] = args.job_type
    job.attrib['CM_VER']    = 'N/A'
    
    var_to_delete_list = []

    dict_of_vars = get_var_dict(job)
    if (dict_of_vars.get("%%ODATE") == None):
        dict_of_vars["%%ODATE"] =  "%%ODATE"

    add_var(job, '%%UCM-RESTART','unchecked')
    add_var(job, '%%UCM-ABENDCODES','Ignore')
    add_var(job, '%%UCM-PREVCONDS','Ignore')
    add_var(job, '%%UCM-AUTOADJUST','Ignore')
    add_var(job, '%%UCM-MFUCC11','Ignore')
    add_var(job, '%%UCM-REJOBID','')
    add_var(job, '%%UCM-MODJCLRESTART','unchecked')
    add_var(job, '%%UCM-APP_NAME',args.job_type)
    add_var(job, '%%UCM-BC_COUNTER','3')
    add_var(job, '%%UCM-JOBORDERID','%%ORDERID')
    add_var(job, '%%UCM-JOBRUNCOUNT','%%RUNCOUNT')
    try:
        add_var(job, '%%UCM-ACCOUNT', job.attrib['OWNER'])
    except KeyError:
        add_var(job, '%%UCM-ACCOUNT', job.attrib['RUN_AS'])
    add_var(job, '%%UCM-PDS', job.attrib["MEMLIB"])
    add_var(job, '%%UCM-JCL', job.attrib["MEMNAME"])

    job.attrib["RUN_AS"] = job.attrib["MEMNAME"]



    del job.attrib['MEMLIB']
    # del job.attrib["OWNER"] ???

    for var in var_to_delete_list:
        del_var(job, var)
    
    return

def convert_job_to_ds(job, args, dict_of_jobnames):

    convert_shout_message(job, args)
    convert_condition_date(job, args)
    convert_condition_logic(job)
    convert_on(job, args)
    convert_output_option(job, args)
    warn_if_attribute_exists_and_remove("job", job.attrib["MEMNAME"], job, "TO_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("job", job.attrib["MEMNAME"], job, "FROM_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("job", job.attrib["MEMNAME"], job, "DUE_OUT_DAYSOFFSET")
    warn_if_attribute_exists_and_remove("job", job.attrib["MEMNAME"], job, "PREV_DAY")
    
    jobname = job.attrib["MEMNAME"].replace("$","_")
    if args.unique.upper() == "Y":
        if jobname not in dict_of_jobnames:
            dict_of_jobnames[jobname] = 0
        else:
            counter = dict_of_jobnames[jobname] + 1
            dict_of_jobnames[jobname] = counter
            jobname = jobname + "_" + str(counter)
    job.attrib['JOBNAME'] = jobname
    del job.attrib['MEMNAME']

    try:
        user = job.attrib["CREATED_BY"]
    except:
        job.attrib["CREATED_BY"] = os.environ.get('USERNAME')
    job.attrib["CHANGE_USERID"] = os.environ.get('USERNAME')
    now = datetime.now()
    job.attrib["CREATION_TIME"] = now.strftime("%H.%M.%S")
    job.attrib["CREATION_DATE"] = now.strftime("%Y%m%d")
    
    if args.nodeid == '':
        host = args.controlm
    else:
        host = args.nodeid
    job.attrib["NODEID"] = host


    return

def get_overlib(job):
    overlib:str = ''
    try:
        overlib = job.attrib['OVERRIDE_PATH']
    except KeyError:
        pass
    try:
        overlib = job.attrib['OVERLIB']
    except KeyError:
        pass
    return overlib

def clear_overlib(job):
    try:
        overlib = job.attrib['OVERRIDE_PATH']
        if overlib != "":
            job.attrib['OVERRIDE_PATH'] = ""
    except KeyError:
        pass
    try:
        overlib = job.attrib['OVERLIB']
        if overlib != "":
            job.attrib['OVERLIB'] = ""
    except KeyError:
        pass
    
    return

def convert_output_option(job, args):
    try:
        if job.attrib["OPTION"] == "Release":
            del job.attrib["OPTION"]
    except KeyError:
        pass
    return

def convert_on(job, args):
    on_to_delete_list = []    
    on_to_append_list = []
    for child in job:
        if child.tag.find("ON") == 0:
            if (child.attrib["PGMS"] != "ANYSTEP"):
                logging.warning(get_time_stamp() + 'In job {} Only ANYSTEP is supported. Removing ON {}'.format(job.attrib["MEMNAME"], ET.tostring(child)))
                on_to_delete_list.append(ET.tostring(child))
            else:
                all_codes = child.attrib["CODE"]
                # .startswith("C")):
                code_list = all_codes.split(",")
                for code in code_list:
                    rel = "EQ"
                    if (code.startswith("<")):
                        rel = "LT"
                        code = code[1:]
                    if (code.startswith(">")):
                        rel = "GT"
                        code = code[1:]
                    if (code.startswith("&lt;")):
                        rel = "LT"
                        code = code[4:]
                    if (code.startswith("&gt;")):
                        rel = "GT"
                        code = code[4:]
                    on = copy.deepcopy(child)
                    del on.attrib["PGMS"]
                    if (code.startswith("C")):
                        code = code[1:]
                    if (code == "EXDER"):
                        code = "NOTOK"
                    if (code != "OK" and code != "NOTOK" and not code.isnumeric()):
                        logging.warning(get_time_stamp() + 'In job {} Only completion code is supported. Removing ON {}'.format(job.attrib["MEMNAME"], ET.tostring(child)))
                        if (ET.tostring(child) not in on_to_delete_list):
                            on_to_delete_list.append(ET.tostring(child))
                        continue
                    if (code.isnumeric()):
                        code = "COMPSTAT " + rel + " " + code
                    on.attrib["CODE"] = code
                    on.attrib["STMT"] = "*"
                    if (validate_do(job, on) > 0):
                        on_to_append_list.append(on)
                    else:
                        logging.info(get_time_stamp() + 'In job {} ON has no valid DO.  Removing ON {}'.format(job.attrib["MEMNAME"], ET.tostring(child)))
                    if (ET.tostring(child) not in on_to_delete_list):
                        on_to_delete_list.append(ET.tostring(child))
 
    while (len(on_to_delete_list) > 0):
           for child in job:
                if ET.tostring(child) in on_to_delete_list:
                    on_to_delete_list.remove(ET.tostring(child))
                    job.remove(child)
    for on in on_to_append_list:
        job.append(on)

    return

def validate_do(job, on):
    do_to_delete_list = []
    do_count = 0
    for child in on:
        do_count = do_count + 1
        if (child.tag.find("DOACTION") == 0 or child.tag == "DO"):
            if (child.attrib["ACTION"] not in ["OK", "NOTOK", "RERUN"]):
                logging.info(get_time_stamp() + 'In job {}, DO ACTION {} not supported and removed'.format(job.attrib["MEMNAME"], ET.tostring(child)))
                do_to_delete_list.append(ET.tostring(child))
        elif child.tag.find("DOOUTPUT") == 0:
            logging.info(get_time_stamp() + 'In job {}, DO OUTPUT {} non supported and removed'.format(job.attrib["MEMNAME"], ET.tostring(child)))
            do_to_delete_list.append(ET.tostring(child))
        elif child.tag.find("DOCOND") == 0:
            i = 5
        elif child.tag.find("DOMAIL") == 0:
            i = 5
        else:
            logging.info(get_time_stamp() + 'In job {}, DO {} non supported and removed'.format(job.attrib["MEMNAME"], ET.tostring(child)))
            do_to_delete_list.append(ET.tostring(child))
    do_left = do_count - len(do_to_delete_list)
    while (len(do_to_delete_list) > 0):
           for child in on:
                if ET.tostring(child) in do_to_delete_list:
                    do_to_delete_list.remove(ET.tostring(child))
                    on.remove(child)
    return do_left

def convert_shout_message(job, args):
    for child in job:
        if child.tag.find("SHOUT") == 0:
            warn_if_attribute_exists_and_remove("Shout " + child.attrib["WHEN"] + " of job", job.attrib["MEMNAME"], child, "DAYSOFFSET")
    return

def convert_condition_date(entity, args):
    if args.cond_days_offset == 'Y':
        return
    entity_type = entity.tag
    if entity_type == "JOB":
        entity_name = entity.attrib["MEMNAME"]
    else:
        entity_name = entity.attrib["FOLDER_NAME"]
    for child in entity:
        if child.tag.find("INCOND") == 0 or child.tag.find("OUTCOND") == 0:
            if child.attrib["ODATE"].startswith("-") or child.attrib["ODATE"].startswith("+"):
                if child.attrib["ODATE"].startswith("-"):
                    child.attrib["ODATE"] = "PREV"
                if child.attrib["ODATE"].startswith("+"):
                    child.attrib["ODATE"] = "NEXT"
                logging.warning(get_time_stamp() + 
                             'In {} {} {} {} with ODATE {} not supported and was changed to PREV/NEXT'
                             .format(entity_type, entity_name, child.tag, child.attrib["NAME"], child.attrib["ODATE"]))
    return


def convert_condition_logic(entity):
    entity_type = entity.tag
    if entity_type == "JOB":
        entity_name = entity.attrib["MEMNAME"]
    else:
        entity_name = entity.attrib["FOLDER_NAME"]
    logic_change:bool = False
    for child in entity:
        next_and_or: str = 'A'
        if child.tag.find("INCOND") == 0:
            and_or:str = next_and_or
            next_and_or = 'A'
            op: str = ''
            try:
                op = child.attrib["OP"]
                if op == "A(":
                    child.attrib["OP"] = "("
                    and_or = 'O'
                    next_and_or = 'A'
                    logic_change = True
                else:
                    next_and_or = 'O'
                child.attrib["AND_OR"] = and_or
            except KeyError:
                pass
    if logic_change:
        logging.warning(get_time_stamp() + 'In {} {}, logic of INCOND relations were manipulated. Please review'.format(entity_type, entity_name))
    return

def add_var(job, name, value):
    logging.debug(get_time_stamp() + 'Adding variable {} with value {}'.format(name, value))
    var = ET.Element('VARIABLE')
    var.attrib['NAME'] = name
    var.attrib['VALUE'] = value
    job.append(var)
    
    # attrib = {}
    # element = job.makeelement('VARIABLE', attrib)
    # job.append(element)
    # element.attrib['NAME'] = name
    # element.attrib['VALUE'] = value


def get_var_content(job, name):
    var_value = ''
    try:
        for var in job.iter('VARIABLE'):
            if var.attrib['NAME'] == name:
                var_value = var.attrib['VALUE']
                break
    except KeyError:
        var_value = ' '
        logging.debug(get_time_stamp() + 'Job {} has variable {} with no value'.format(job.attrib['JOBNAME'], name))
        print('Job {} has variable {} with no value'.format(job.attrib['JOBNAME'], name))
    return var_value

def set_var_content(job, name, value):
    for var in job.iter('VARIABLE'):
        if var.attrib['NAME'] == name:
            var.attrib['VALUE'] = value
            break
    return

def del_var(job, var_name):
    for var in job.iter('VARIABLE'):
        if var.attrib['NAME'] == var_name:
            job.remove(var)
    return

def get_var_dict(job):
    dict_of_vars:dict = {}
    for var in job.iter('VARIABLE'):
        try:
            dict_of_vars[var.attrib['NAME']] = var.attrib['VALUE']
        except KeyError:
            dict_of_vars[var.attrib['NAME']] = ''

    return dict_of_vars

def generate_log_file(log):
    dirpath = os.path.join("log")
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "mainframe-xml-to-ds.log")
    if os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.ERROR)
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.WARNING)
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.INFO)
    elif log == 3:
        logging.basicConfig(filename=path, level=logging.DEBUG)

def get_time_stamp():
    return str(datetime.now()) + " "

if __name__ == "__main__":
    start = time.time()
    main()
    end = time.time()
    print("It took:", math.floor((end-start)/60), "minutes and", math.floor((end-start)%60), "seconds")