import xml.etree.ElementTree as ET
import csv
import argparse
import logging
from datetime import datetime
import os
import ctypes  # An included library with Python install.   

def argument_parser():
    parser = argparse.ArgumentParser(
        description="Create a CSV folder/resource xref from XML")
    parser.add_argument('-i', dest='input_file', type=str,
                        help='Input XML file', metavar='', required=True)
    parser.add_argument('-o', dest='output_file', type=str,
                        help='Output CSV file', metavar='', required=True)
    parser.add_argument('-log', dest='log', type=int,
                        help=argparse.SUPPRESS, default=0, metavar='')
    return parser

def build_res_dict(root, folder_type, dictionary):
    for folder in root.iter(folder_type):
        folder_name = folder.attrib.get("FOLDER_NAME", "")
        server = folder.attrib.get("DATACENTER")
        # build a list of all quantitve and control res at job level
        for job in folder.findall("JOB"):
            for res in job.findall("QUANTITATIVE"):
                resname = res.attrib.get("NAME")
                quant = res.attrib.get("QUANT")
                key = (resname, folder_name, server)
                value = ("QUANTITATIVE", quant)
                dict_add(key,value,dictionary)
            for res in job.findall("CONTROL"):
                resname = res.attrib.get("NAME")
                type = res.attrib.get("TYPE")
                key = (resname, folder_name, server)
                value = ("CONTROL", type)
                dict_add(key,value,dictionary)
        # add folder level resources
        for res in folder.findall("QUANTITATIVE"):
            resname = res.attrib.get("NAME")
            quant = res.attrib.get("QUANT")
            key = (resname, folder_name, server)
            value = ("QUANTITATIVE", quant)
            dict_add(key,value,dictionary)
            
        for res in folder.findall("CONTROL"):
            resname = res.attrib.get("NAME")
            type = res.attrib.get("TYPE")
            key = (resname, folder_name, server)
            value = ("CONTROL", type)
            dict_add(key,value,dictionary)
        
def dict_add(key,value,dictionary):
    if key not in dictionary:
        dictionary[key] = value
        
def write_xref(writer,res_dict):
    iter=1
    for entry in sorted(res_dict.items()):
        (resname, folder_name, server), (restype,attrib) = entry
        writer.writerow([resname,restype,folder_name,server])

def generate_log_file(log):
    dirpath = "log"
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "cond_folder_xref.log")
    if os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.INFO)
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.ERROR)
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.DEBUG)

def get_time_stamp():
    return str(datetime.now()) + " "

def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info(get_time_stamp() + ' script parameters: ' + str(args))
    
    # I'm using dictionary because it should be more performing for key/value searching
    folders={}
  
    print("step 0 - parsing xml file",args.input_file, end='\r' )
    logging.info(get_time_stamp() + ' parsing xml file')
    tree = ET.parse(args.input_file)
    root = tree.getroot()
    
    print("step 1 - processing folders for conditions search", end='\r' )
    
    logging.info(get_time_stamp() + ' building smart folders dictionary')    
    build_res_dict(root, "SMART_FOLDER", folders)
    
    logging.info(get_time_stamp() + ' building "normal" folders dictionary')        
    build_res_dict(root, "FOLDER", folders)
    
    print("step 2 - creating csv XREF                             ", end='\r' )
    logging.info(get_time_stamp() + ' starting CSV creation')        
    done = 'N'
    while done != 'Y':
        try:
            with open(args.output_file, 'w', newline='') as csv_file:
                csv_writer = csv.writer(csv_file)
                csv_writer.writerow(['Resource Name','Resource Type','Folder Name','Control-M Server Name'])
                write_xref(csv_writer,folders)
                done = 'Y'
        except PermissionError:
            button = ctypes.windll.user32.MessageBoxW(0, "Please close the XREF report", "Warning. File open in Excel", 5)
            if button!=4:
                done = 'Y'
        except:
            ctypes.windll.user32.MessageBoxW(0, "Sorry, something went wrong - see output", "Error writing XREF report", 0)
            raise

    print(f"Data extracted from {args.input_file} and saved to {args.output_file}")
    logging.info(get_time_stamp() + ' all done')        

if __name__ == "__main__":
    main()
    