from folder import folder
from folder_group import folder_group
from folder_fetcher import folder_fetcher


#
# Folders Merger
# -------------- 
# This is the main logic of the workflow breakers. 
# It breaks a set of folders (main workflow read from the file) to several distinct sets of folders (distinct workflows) that have no resource in common. 
#
class folders_merger:

    def __init__(self, file: folder_fetcher) -> list:
        self._file_ = file

    def merge_folders_to_groups(self):
        # The routine maintains a list of folder groups (all the folders in each group have at least one resource in common with at least one other folder in the group).
        # This routine reads all the folders and their data (folder objects) one after the other using the "fetcher".
        # For each folder the routine gets from the "fetcher":
        #    + All the groups in the list are scanned and marked if they are intersecting with the folder (have at least one common resource with it)
        #    + New group is created and the new folder is added to it (as the only folder in the group)
        #    + All the marked groups are merged with the new group (all the intersecting groups are merged with the new group that has the new folder)
        #    + All the groups that were merged to the new group are removed from the list and deleted
        #    + The new group is added to the list
        #    + Once all the folders are read, the group names are adjusted so they will be in sequance (groups were removed so the sequance is lost)
        #

        grp_list = []  # We start with an empty group list

        #
        # The main loop that reads the folders from the file using the "fetcher" and process them
        #
        f: folder = self._file_.get_folder()  # Read first folder
        while (f != None):  # While there are still folders in the file...
            print(' ' * 80, end='\r')  # clear for spinner message...
            print('Processing folder ' + f.folder_name() + '...', end='\r')  # Spinner message...
            self._mark_intersecting_groups_(grp_list, f)  # Mark all the groups intersecting with the folder in the list
            g = folder_group(f)  # Create a new group with the just read folder
            del f  # Delete the folder (already "copied" to the group)
            self._merge_groups_(grp_list, g)  # Merge the marked groups with the new group
            self._delete_merged_groups_(grp_list)  # Delete all merged groups and remove them from the groups list
            grp_list.append(g)  # Add the new just created group to the list
            f = self._file_.get_folder()  # Read next folder from the file

        self._adjust_groups_names_(grp_list)  # Adjust the group names so they will be in sequance

        return grp_list

    def _mark_intersecting_groups_(self, grp_list, f):
        #
        # Mark all the groups in the given list that intersect with the given folder (have at least one common resource)
        #
        for group in grp_list:
            if (group.res_intersects_with(f)):
                group.set_need_merge(True)

    def _merge_groups_(self, grp_list, g):
        #
        # Merge all the groups that are marked as "need to be merged" with the group that incldues the new folder
        #
        for group in grp_list:
            if group.need_merge():
                g.merge_with(group)

    def _delete_merged_groups_(self, grp_list):
        #
        # delete all teh groups that were already meged and remove them from the groups list
        #
        all_deleted = False
        while (all_deleted == False):
            all_deleted = True
            for group in grp_list:
                if group.need_merge():
                    grp_list.remove(group)
                    del group
                    all_deleted = False
                    break  # must break the "for loop" (list iterator) after an element is removed and restart the "while" iteration!!!

    def _adjust_groups_names_(self, grp_list):
        #
        # Sort the list by the jobs count and adjust the group names as they are no longer in sequance
        #
        grp_list.sort(key=lambda folder_group: folder_group.jobs_count(), reverse=True)

        i = 1
        for group in grp_list:
            group.set_group_name('Grp-' + str(i).zfill(10))
            i = i + 1
