#!./perl



#---------------------------------------------------------------------#
#           Utility CTMJRBC                                           #
#           ---------------                                           #
#                                                                     #
# This utility changes the scheduling criteria from jobs to general   #
# RBCs that can be shared by similar jobs.                            #
#                                                                     #
# Default input file:           input.xml                             #
#                                                                     #
# Output files are:             output.xml                            #
#                                                                     #
#---------------------------------------------------------------------#
#---------------------------------------------------------------------#
# Created          : DEC 1, 2010                                      #
# Updated          :                                                  #
# Programer : Yakov Steinmetz                                         #
# Team      : CONTROL-M for z/OS, Tel-Aviv, BMC Software              #
#---------------------------------------------------------------------#

use XML::DOM;
use String::ProgressBar;

# Turn on autoflushing as by default output is buffered until a newline is seen.
# This is a problem because the progress bar needs to be displayed constantly.
use IO::Handle;
STDERR->autoflush(1);
STDOUT->autoflush(1);

 
#---------------------------------------------------------------------#
# Initializing Constants                                              #
#---------------------------------------------------------------------#

$FALSE = 0;
$TRUE  = 1;

$Utility_name   = 'CTMJRBC';
$V630		= '630';
$V700  		= '700';
$Group_table	= 'Group';
$SMART_table 	= 'SMART';
$Regular_table	= 'Regular';

$Empty_value  = '##EMPTY##';			       				# Indicates that there is no value 

$JOB_element 			    = 'JOB';	                      	        # All releases - XML JOB element name  
$V700_SMART_table_start_element     = 'SMART_TABLE';   				# 700 SMART table
$V700_regular_table_start_element   = 'TABLE';         				# 700 regular table
$SUB_TABLE_element 		    = 'SUB_TABLE';              		# 700 and higher - XML SUB_TABLE element name  
$V700_RBC_in_table_element_name	    = 'RULE_BASED_CALENDAR';
$V700_RBC_in_job_element_name       = 'RULE_BASED_CALENDARS';
$V700_RBC_name_in_job_element	    = 'NAME';          				# <RULE_BASED_CALENDARS NAME="xxx"
$V700_RBC_relationship_job_element  = "RULE_BASED_CALENDAR_RELATIONSHIP";       # RULE_BASED_CALENDAR_RELATIONSHIP="AND" or "OR"

$V630_group_table_start_element     = 'SCHED_GROUP';   				# 630 group table  
$V630_regular_table_start_element   = 'SCHED_TABLE';   				# 630 regular table  
$V630_TAG_in_table_element_name	    = 'TAG';
$V630_TAG_in_job_element_name	    = 'TAG_NAMES';
$V630_TAG_name_in_job_element	    = 'TAG_NAME';      				# <TAG_NAMES   TAG_NAME="xxx"
$V630_TAG_relationship_job_element  = 'TAG_RELATIONSHIP';			# TAG_RELATIONSHIP="AND" or "OR"

$Adjust_condition                   = 'N';

$Default_RBC_prefix                 = 'GEN_RBC';
$Default_TAG_prefix                 = 'GEN_TAG';
$RBC_TAG_group_prefix               = '#GROUP#';
$RBC_TAG_table_prefix               = '#TABLE#';
$RBC_user_prefix                    = '';  # Assume that the user did not specify a RBC/TAG prefix
$Saved_group_name                   = '';  # The saved group name  
$Saved_table_name                   = '';  # The saved table name
$Saved_memname_jobname              = '';  # The saved memname (z/OS) or job name (EM)
$Release			    = '';  # Will be set to 630/700
$RBC_TAG_name_seq                   = 0;   # The sequence number when generating an RBC/TAG name from the table name or group name or a user-specified prefix  


$Default_owner                      = 'CTM_OWNER';
$Owner                              = $Default_owner;

$Default_maxwait                    = '00';
$Default_priority                   = '00';
$Default_application                = 'NO_APPL';

$Definition_active_from             = 'ACTIVE_FROM';
$Definition_active_until            = 'ACTIVE_TILL';
$Saved_definition_active_from       = '';
$Saved_definition_active_until      = '';   

$Weekdays                           = 'WEEKDAYS';

$zOS_job_MEMNAME     = 'MEMNAME';
$EM_job_JOBNAME      = 'JOBNAME';
$Job_memname_jobname = '';        # This will be set to MEMNAME or JOBNAME
$XML_is_EM      = $TRUE;          # Assume that this is an EM XML (job parameter has the JOBNAME parameter)

# A list job parameters that will be used to create an RBC.  These parameters will be deleted from the job, except for MAXWAIT and RETRO.
$MAXWAIT = 'MAXWAIT';
$RETRO   = 'RETRO';
@V630_job_RBC_scheduling_parameters_list = qw(MAXWAIT DAYS DAYS_AND_OR WEEKDAYS DATE DAYSCAL WEEKSCAL CONFCAL RETRO SHIFT SHIFTNUM JAN FEB MAR APR MAY JUN JUL AUG SEP OCT NOV DEC);
@V700_job_RBC_scheduling_parameters_list = qw(MAXWAIT DAYS DAYS_AND_OR WEEKDAYS DATE DAYSCAL WEEKSCAL CONFCAL RETRO SHIFT SHIFTNUM JAN FEB MAR APR MAY JUN JUL AUG SEP OCT NOV DEC);
@Months_list = qw(JAN FEB MAR APR MAY JUN JUL AUG SEP OCT NOV DEC);

@Adjust_condition_list = qw(Y N Dummy);   # Valid values for the ADJUST_COND parameter

$XML_Debug_Print = 'Debug_XML.TXT';


#---------------------------------------------------------------------#
# Initializing Variables & Command-Line Flags                         #
#---------------------------------------------------------------------#

$Input_XML_file	    = 'input.xml';

$Output_XML_file    = 'output.xml';

my $dom_parser = new XML::DOM::Parser (ErrorContext => 1);	# Create a parser object, specify to display line in error if XML not well formed

$XML_chunk = '';			            # Will be used to store a 'chunk' (one table/group) of the parsed XML
    
$JA_p    	                 = 0;     	# Pointer to the job array
@jobArray                    = '';    	# This array is used to save the scheduling parameters needed to to build an RBC/TAG for all jobs in a table
$JA_scheduling_parms         =  0;      # Array of scheduling parameters
$JA_array_RBC_TAG_name       =  1;      # The name of the RBC/TAG

$RBC_p    	                 = 0;     	# Pointer to the RBC array
@RBCarray                    = '';    	# This array is used to save the scheduling parameters of all RBCs
$RBC_scheduling_parms        =  0;      # Array of scheduling parameters
$RBC_array_TAG_name          =  1;      # The name of the RBC/TAG

$Unique_RBCs_TAGs_p    	     = 0;     	# Pointer to the unique RBCs/TAGs array
@Unique_RBCs_TAGs_list       = '';      # List of unique RBCs/TAGs

$No_tables_with_SUB_TABLE    = 0;
$No_jobs_processed           = 0;
$No_jobs_with_RBC_TAG        = 0;
$No_jobs_modified            = 0;
$No_tables_modified          = 0;

use Cwd;

$Directory_Name = getcwd();    #  The default directory name is the working directory

while ($ARGV[0])
{
  if ($ARGV[0] eq '-DIRECTORY')
  {
    shift(@ARGV);
    $Directory_Name = $ARGV[0];
    shift(@ARGV);
  }
  elsif ($ARGV[0] eq '-INPXML')
  {
    shift(@ARGV);
    $Input_XML_file = $ARGV[0];
    shift(@ARGV);
  }
  elsif ($ARGV[0] eq '-OUTXML')
  {
    shift(@ARGV);
    $Output_XML_file = $ARGV[0];
    shift(@ARGV);
  }
  elsif ($ARGV[0] eq '-ADJUST_COND')
  {
    shift(@ARGV);
    $Adjust_condition = $ARGV[0];
    shift(@ARGV);
  }
  elsif ($ARGV[0] eq '-RBC_TAG_PREFIX')
  {
    shift(@ARGV);
    $RBC_user_prefix = $ARGV[0];
    shift(@ARGV);
  }
  elsif ($ARGV[0] eq '-OWNER')
  {
    shift(@ARGV);
    $Owner = $ARGV[0];
    shift(@ARGV);
  }  
  else
  {
    print  ("\nInvalid Argument: ",$ARGV[0],"\n\n");
    print  ("Usage: convert\n");
    print  ("     [-DIRECTORY <def directory   > ] {Directory where the XML files are present. Default: current directory}\n");
    print  ("     [-INPXML    <input XML file  > ] {Name of the input XML file. Default: input.xml}\n\n");
    print  ("     [-OUTXML    <output XML file > ] {Name of the output XML file. Default: output.xml}\n\n");
    print  ("     [-ADJUST_COND  <Y N Dummy> ] {Whether to set the XML adjust condition parameter.    Default: N  }\n\n");
    print  ("     [-OWNER  <owner> ] {The owner - refer to CONTROL-M User Guide.  Default: NO_OWNER}\n\n");    
    die    ("     [-RBC_TAG_PREFIX   <#GROUP# #TABLE# user-specified-prefix> ] {Whether to use the group name, table name, or user specified prefix for RBC/TAG names. Default: GEN_RBC (700 or later) or GEN_TAG (630)}\n\n");          
  }


}



  #----------------------------------------------------#
  #  Add the path to the file names                    #
  #----------------------------------------------------#

  # Check for special output XML file directives
  if ($Output_XML_file eq '##PREFIX##')
  {
    $Output_XML_file = 'OUT.'.$Input_XML_file;
  }

  if ($Output_XML_file eq '##SUFFIX##')
  {
    $Output_XML_file = Change_XML_output_file_name($Input_XML_file);
  }

  $Input_XML_file = $Directory_Name . '/' . $Input_XML_file;

  $Output_XML_file = $Directory_Name . '/' . $Output_XML_file;
 
  $XML_Debug_Print = $Directory_Name . '/' . $XML_Debug_Print;
  
  #Open the input XML file
  unless(open($XML_input_file, $Input_XML_file))
  {
    die("Could not open input XML file: ".$Input_XML_file.".    Aborting..\n");
  }

 
  #---------------------------------------------------------------------#
  # Creating the Output files                                           #
  #---------------------------------------------------------------------#
  unless(open($XML_output_file, "> $Output_XML_file"))
  {
    die("Could not create ", $Output_XML_file, " file. Aborting..\n");
  }

  Debug_print_XML_buffer("open");
 
  unless (open($Exceptions_file,"> exceptions.out"))
  {
    die ("Could not create exceptions.out file. Aborting..\n");
  }

  unless (open($Progress_file,"> progress.out"))
  {
    die ("Could not create progress.out file. Aborting..\n");
  }

  unless (open($changeList_file,"> change_list.out"))
  {
    die ("Could not create change_list.out file. Aborting..\n");
  }

  #Check the ADJUST_COND parameter
  if (!Parm_on_list($Adjust_condition,@Adjust_condition_list))   # Is it a valid value?
  {
    print $Exceptions_file "The ADJUST_COND parameter value ".$Adjust_condition." is not valid.  Valid values are Y, N, and Dummy\n";
    print $Exceptions_file "The parameter was set to the default value of N\n";
    $Adjust_condition = 'N';
  }
  
  # Check the RBC_TAG_PREFIX parameter
  if (!There_is_a_value($RBC_user_prefix))
  {
    $RBC_user_prefix = '';
  }

  # The RBC_TAG_PREFIX parameter can't be longer than 15 characters
  if (length($Temp) > 15)
  {
    print $Exceptions_file "The RBC_TAG_PREFIX parameter ".$RBC_user_prefix." is longer than 15 characters\n";
    print $Exceptions_file "The parameter was set to the default value of ".$Default_RBC_prefix." or ".$Default_TAG_prefix."\n\n";
    $RBC_user_prefix = '';
  }
  
  $Parm_RBC_TAG_prefix = $RBC_user_prefix;   # Assume that the user entered an RBC/TAG prefix  

  # Check the OWNER parameter
  if (!There_is_a_value($Owner))
  {
    $Owner = $Default_owner;
  }
 


  my $filesize = -s $Input_XML_file;                    # file size
  my $formatted_filesize = niceSize($filesize,1);	# get the formatted file size - "size "nn.n" (one decimal) will be displayed
  my $fileName = "Input XML file: ".$Input_XML_file." ( ".$formatted_filesize." )\n\n";

  $Msg_start = "\n\nUtility ".$Utility_name." Version 1.0\n\n";
  $Msg_time  = "Starting process at: ".scalar(localtime(time()))."\n\n";
  
  print $Exceptions_file $Msg_start;
  print $Exceptions_file $Msg_time;
  print $Exceptions_file $fileName;
  
  print $changeList_file $Msg_start;
  print $changeList_file $Msg_time;
  print $changeList_file $fileName;

  
  print $Progress_file $Msg_start;
  print $Progress_file $Msg_time;
  print $Progress_file $fileName;
  
  print $Msg_start;
  print $Msg_time;
  print $fileName;
  
  $Total_tables_progress      = 0;   
  $Curr_table_number_progress = 0;   
  
  # Count the number of tables(for 700) / groups (for 630)
  while ($Count_line = <$XML_input_file> )
  {
    $Temp = join(' ', split(' ', $Count_line));                                 #Remove leading and trailing blanks

    if (substr($Temp, 0, 12) eq '<'.$V700_SMART_table_start_element)   		# Is this a 700 SMART table?
    {
      $Total_tables_progress++;
    }
  
    elsif (substr($Temp, 0, 6) eq '<'.$V700_regular_table_start_element)       	# Is this a 700 regular table (not a SMART table)?
    {
      $Total_tables_progress++; 
    }
    elsif (substr($Temp, 0, 12) eq '<'.$V630_group_table_start_element)		# Is this a 630 group table?
    {
      $Total_tables_progress++; 
    }
    elsif (substr($Temp, 0, 12) eq '<'.$V630_regular_table_start_element)	# Is this a 630 regular (not group) table?
    {
     $Total_tables_progress++;
    }
  }


  $progress = String::ProgressBar->new( max => $Total_tables_progress);
  
  $No_times_to_display_progress = 100;  # A progress bar will be displayed each time an additional 1% of the tables were processed
  $Display_progress_after_this_count  = (int($Total_tables_progress/$No_times_to_display_progress)) + 1;   # Display the progress every time it reaches this number
  
  #When $Current_progress_count reaches $Display_progress_after_this_count, a progress message is displayed, and $Current_progress_count is reset to zero.    
  $Current_progress_count       = 0;   

  close $XML_input_file;

  #re-open the input XML file
  unless(open($XML_input_file, $Input_XML_file))
  {
    die("Could not open input XML file: ".$Input_XML_file.".    Aborting..\n");
  }


# Write all the XML lines up to DEFTABLE
while (my $XML_line = <$XML_input_file> )
{
  $Temp = join(' ', split(' ', $XML_line));		#Remove leading and trailing blanks

  print $XML_output_file $XML_line;

  last if (substr($Temp, 0, 9) eq '<DEFTABLE');
}


# Process all the tables(for 700) / groups (for 630)   
while (my $XML_line = <$XML_input_file> )
{

  # The next XML record indicates if this is a group (630) / SMART table (700), or a regular table
  # For 700:
  #    <SMART_TABLE>   SMART table 
  #    <TABLE>         regular table 
  # For 630:
  #    <SCHED_GROUP>   group table
  #    <SCHED_TABLE>   regular table


  $Temp = join(' ', split(' ', $XML_line));#Remove leading and trailing blanks

  if (length($Temp) eq 0)    # Skip empty lines
  {
    next;    
  }
  
  $XML_in_memory = $XML_line;

  if (substr($Temp, 0, 12) eq '<'.$V700_SMART_table_start_element)   			# Is this a 700 SMART table?
  {
    $Parm_release 	  		    = $V700;					# Release 700
    $Parm_Table_type 	  		    = $SMART_table;				# This is SMART table
    $Parm_start_table_element	  	    = $V700_SMART_table_start_element; 		# The start element for SMART tables
    $Parm_non_regular_table_changed_element = $V700_SMART_table_start_element;		# Only relevant for regular tables 
    $Parm_End_element	  		    = '</'.$V700_SMART_table_start_element; 	# The end element for SMART tables
    $Parm_End_element_len 		    = 13;					# SMART table end element length
    $Parm_RBC_TAG_in_table_element_name     = $V700_RBC_in_table_element_name;		# RBC element in a table
    $Parm_RBC_TAG_in_job_element_name       = $V700_RBC_in_job_element_name;		# RBC element in a job
    $Parm_RBC_TAG_name_in_job_element       = $V700_RBC_name_in_job_element;            # NAME= in <RULE_BASED_CALENDARS NAME="xxx"
    @Parm_job_scheduling_parameters_list    = @V700_job_RBC_scheduling_parameters_list;	# List of scheduling parameters to be used when adding an RBC
    $Parm_no_scheduling_parameters = (scalar @Parm_job_scheduling_parameters_list) - 1; # Number of items in this array 
    $Parm_max_RBC_TAG_length	  	    = 20; 		                        # The max length of an RBC/TAG in 700
    $Parm_RBC_TAG_no_group_prefix           = $Default_RBC_prefix;                      # If #GROUP# was specified, but there is no group value
    $Parm_RBC_TAG_relationship_job_element  = $V700_RBC_relationship_job_element;  
 				
    if ($Parm_RBC_TAG_prefix eq '')
    {
      $Parm_RBC_TAG_prefix = $Default_RBC_prefix;
    }
  }
  elsif (substr($Temp, 0, 6) eq '<'.$V700_regular_table_start_element)			# Is this a 700 regular table (not a SMART table)?
  {
    $Parm_release 	  		    = $V700;					# Release 700   
    $Parm_Table_type 	  		    = $Regular_table;				# This is a regular table (not SMART)
    $Parm_start_table_element 		    = $V700_regular_table_start_element;	# The start element for regular tables
    $Parm_non_regular_table_changed_element = $V700_SMART_table_start_element;		# Change "TABLE" to this value ("SMART_TABLE") 
    $Parm_End_element 	  		    = '</'.$V700_regular_table_start_element;	# The end element for regular tables
    $Parm_End_element_len 		    = 7;					# Regular table end element length
    $Parm_RBC_TAG_in_table_element_name     = $V700_RBC_in_table_element_name;		# RBC element in a table
    $Parm_RBC_TAG_in_job_element_name	    = $V700_RBC_in_job_element_name;		# RBC element in a job
    $Parm_RBC_TAG_name_in_job_element       = $V700_RBC_name_in_job_element;            # NAME= in <RULE_BASED_CALENDARS NAME="xxx"
    @Parm_job_scheduling_parameters_list    = @V700_job_RBC_scheduling_parameters_list;	# List of scheduling parameters to be used when adding an RBC
    $Parm_no_scheduling_parameters = (scalar @Parm_job_scheduling_parameters_list) - 1; # Number of items in this array 
    $Parm_max_RBC_TAG_length	  	    = 20; 		                        # The max length of an RBC/TAG in 700
    $Parm_RBC_TAG_no_group_prefix           = $Default_RBC_prefix;                      # If #GROUP# was specified, but there is no group value
    $Parm_RBC_TAG_relationship_job_element  = $V700_RBC_relationship_job_element;  
    
    if ($Parm_RBC_TAG_prefix eq '')
    {
      $Parm_RBC_TAG_prefix = $Default_RBC_prefix;
    }
  }
  elsif (substr($Temp, 0, 12) eq '<'.$V630_group_table_start_element)			# Is this a 630 group table?
  {
    $Parm_release 	  		    = $V630;					# Release 630
    $Parm_Table_type           		    = $Group_table;				# This is a group table
    $Parm_start_table_element               = $V630_group_table_start_element;		# The start element for group tables
    $Parm_non_regular_table_changed_element = $V630_group_table_start_element;		# Only relevant for regular tables 
    $Parm_End_element           	    = '</'.$V630_group_table_start_element;	# The end element for group tables
    $Parm_End_element_len 	  	    = 13;					# Group table end element length
    $Parm_RBC_TAG_in_table_element_name     = $V630_TAG_in_table_element_name;		# TAG element in a table 
    $Parm_RBC_TAG_in_job_element_name 	    = $V630_TAG_in_job_element_name;		# TAG element in a job
    $Parm_RBC_TAG_name_in_job_element       = $V630_TAG_name_in_job_element;            # TAG_NAME in <TAG_NAMES   TAG_NAME="xxx"
    @Parm_job_scheduling_parameters_list    = @V630_job_RBC_scheduling_parameters_list;	# List of scheduling parameters to be used when adding an RBC	
    $Parm_no_scheduling_parameters = (scalar @Parm_job_scheduling_parameters_list) - 1; # Number of items in this array 
    $Parm_max_RBC_TAG_length	  	    = 20; 		                        # The max length of an RBC/TAG in 630
    $Parm_RBC_TAG_no_group_prefix           = $Default_TAG_prefix;                      # If #GROUP# was specified, but there is no group value
    $Parm_RBC_TAG_relationship_job_element  = $V630_TAG_relationship_job_element;

    if ($Parm_RBC_TAG_prefix eq '')
    {
      $Parm_RBC_TAG_prefix = $Default_TAG_prefix;
    }
  }
  elsif (substr($Temp, 0, 12) eq '<'.$V630_regular_table_start_element)			# Is this a 630 regular (not group) table?
  {
    $Parm_release 	  		    = $V630;					# Release 630
    $Parm_Table_type           		    = $Regular_table;				# This is a regular (not group) table
    $Parm_start_table_element          	    = $V630_regular_table_start_element;	# The start element for regular tables
    $Parm_non_regular_table_changed_element = $V630_group_table_start_element;		# Change "SCHED_TABLE" to this value ("SCHED_GROUP")
    $Parm_End_element          		    = '</'.$V630_regular_table_start_element;	# The end element for regular tables
    $Parm_End_element_len      		    = 13;					# Regular table end element length
    $Parm_RBC_TAG_in_table_element_name     = $V630_TAG_in_table_element_name;		# TAG element in a table   
    $Parm_RBC_TAG_in_job_element_name 	    = $V630_TAG_in_job_element_name;		# TAG element in a job
    $Parm_RBC_TAG_name_in_job_element       = $V630_TAG_name_in_job_element;            # TAG_NAME in <TAG_NAMES   TAG_NAME="xxx"
    @Parm_job_scheduling_parameters_list    = @V630_job_RBC_scheduling_parameters_list;	# List of scheduling parameters to be used when adding an RBC	
    $Parm_no_scheduling_parameters = (scalar @Parm_job_scheduling_parameters_list) - 1; # Number of items in this array 
    $Parm_max_RBC_TAG_length	  	    = 20; 		                        # The max length of an RBC/TAG in 630
    $Parm_RBC_TAG_no_group_prefix           = $Default_TAG_prefix;                      # If #GROUP# was specified, but there is no group value
    $Parm_RBC_TAG_relationship_job_element  = $V630_TAG_relationship_job_element;
    if ($Parm_RBC_TAG_prefix eq '')
    {
      $Parm_RBC_TAG_prefix = $Default_TAG_prefix;
    }

  }

  $Parm_number_of_records_in_table = 0;
  $Parm_job_number                 = 0;  # for displaying the job number within the table
  
  # Change this group/table
  $record_count = 0;
  while (my $XML_line = <$XML_input_file> )
  {
    #$XML_in_memory = $XML_in_memory.$XML_line;   # very slow!!!
    $XML_in_memory .= $XML_line;
    $Temp = join(' ', split(' ', $XML_line));						# Remove leading and trailing blanks
    $record_count++;
    if (substr($Temp, 0, $Parm_End_element_len) eq $Parm_End_element)			# Is this the table end element?
    {
	Modify_the_Table();
        last;
    }
    else
    {
      $Parm_number_of_records_in_table++;
    }

  } 
}


  print $progress->update($Curr_table_number_progress)->string()."\r";    # The final display
  print $XML_output_file '</DEFTABLE   > ';#Write the last XML line

  

  
  #--------------------------------------------------------------------#
  # Close the files                                                    #
  #--------------------------------------------------------------------#
  close ($XML_input_file);
  close ($XML_output_file);
  Debug_print_XML_buffer('close');
 
  $End_msg = "\n\nEnd of utility ".$Utility_name." at: ".scalar(localtime(time()))."\n";
  print $Exceptions_file $End_msg;
  print $changeList_file "\n\n";
  print $changeList_file "Number of tables processed:                ".$Curr_table_number_progress."\n";
  print $changeList_file "Number of tables processed with SUB_TABLE: ".$No_tables_with_SUB_TABLE."\n";
  print $changeList_file "Number of jobs processed:                  ".$No_jobs_processed."\n";
  print $changeList_file "Number of jobs processed with RBC/Tag:     ".$No_jobs_with_RBC_TAG."\n";
  print $changeList_file "\n";
  print $changeList_file $End_msg;
  print $Progress_file $End_msg;
  
  close ($Exceptions_file); 
  close ($Progress_file);
  close ($changeList_file);
 
  print $End_msg;
   
  exit;

# Modify this table
# There are a number of tables that are handled by this utility:
#
# 1. CTM 630 regular table:
#
#    <SCHED_TABLE  TABLE_NAME="YAKOV"    DATACENTER="P1MTROLM"             TABLE_DSN="CTMP.V630.SCHEDULE">
#      <JOB
#       MEMNAME="ABEND"
#       DAYS_AND_OR="OR"
#       DEC="1"
#       NOV="1"
#      </JOB>
#      <JOB
#       MEMNAME="IOACND63"
#       DAYS_AND_OR="OR"
#       DEC="1"
#       NOV="1"
#      >
#      </JOB>
#    </SCHED_TABLE>
#
#
# 2. CTM 630 group table:
#
#    <SCHED_GROUP  TABLE_NAME="YAKOVGR3" DATACENTER="P1MTROLM"   TABLE_DSN="CTMP.V630.SCHEDULE"
#      GROUP="YAKOVGR3"
#      MEMNAME="YAKOVGR3"
#      MAXWAIT="00"
#    > 
#    <JOB
#      MEMNAME="JCLERROR"
#      MAXWAIT="00"
#      DEC="1"
#      NOV="1"
#    > 
#    </JOB>
#    <JOB
#    MEMNAME="IEFBR14"
#    MAXWAIT="00"
#    >  
#    </JOB>
#    </SCHED_GROUP>
#
#
# 3. CTM 700 regular table:
#
#    <TABLE TABLE_NAME="YAKOV"    DATACENTER="P1MTROLM"             TABLE_DSN="CTMP.V700.SCHEDULE"                           > 
#      <JOB 
#       MEMNAME="KSL" 
#       MAXWAIT="01" 
#      > 
#      </JOB> 
#      <JOB 
#       MEMNAME="IEFBR14" 
#       JAN="1" 
#      > 
#   </TABLE> 
#
#
# 4. CTM 700 Smart table (without SUB_TABLE):
#
#    <SMART_TABLE TABLE_NAME="YAKOVRBC" DATACENTER="P1MTROLM" TABLE_DSN="CTMP.V700.SCHEDULE" 
#       MEMNAME="YAKOVRBC" 
#       MAXWAIT="00" 
#    > 
#       <RULE_BASED_CALENDAR NAME="EVERY_DAY" 
#         MAXWAIT="00" 
#         DAYS="ALL" 
#         JAN="1" 
#       /> 
#       <JOB 
#         MEMNAME="IEFBR14" 
#         GROUP="NO_GROUP"  
#       > 
#       <RULE_BASED_CALENDARS NAME="EVERY_DAY" /> 
#       </JOB> 
#    </SMART_TABLE> 
#
#
# 5. CTM 700 Smart table (with SUB_TABLE):
#///////////////////// Not supported in this release -- a warning message is issued ///////////////////
#    Note: There may be a SUB_TABLE within a SUB_TABLE!
#
#    If a SUB_TABLE is present, the name of the RBC must be specified in the SUB_TABLE.
#
#	<SMART_TABLE
#	    DATACENTER="SFC-ECOHEN-38"
#	    TABLE_NAME="PT-DWH-PC-A-BSUMM"
#	    APPLICATION="DEFAULT_APPLICATION"
#	>
#	   <TAG
#	     WEEKDAYS="ALL"
#	     NOV="1"
#	   />
#	   <SUB_TABLE
#		ADJUST_COND="0"
#		CREATION_USER="97"
#		APPLICATION="DEFAULT_APPLICATION"
#		JOBNAME="PT-DWH-PC-A-BSUMM_L3BLD"
#		DESCRIPTION=""
#	   >
#		    <TAG_NAMES TAG_NAME="ALL"/>
#			<JOB
#			PARENT_TABLE="PT-DWH-PC-A-BSUMM_L3BLD"
#			MEMNAME="sched.env"
#			MEMLIB="/dwh_app"
#			JOBNAME="PT-DWH-PJ-A-BSUMM_L3BLD_3"
#			DESCRIPTION="Criacao de qualquer coisa"
#			>
#			<TAG_NAMES TAG_NAME="ALL"/>
#			</JOB>
#		</SUB_TABLE>
#   </SMART_TABLE >  
#
sub Modify_the_Table
{
  
  Display_XML_release($Parm_release);
  
  $XML_chunk = $dom_parser -> parse($XML_in_memory); 	# Parse the XML in memory

  #Debug_print_XML_buffer('print',$XML_chunk);	        # For debug--print the XML parser buffer

  Process_the_table();									# Process this table
  
  $XML_output_text = $XML_chunk -> toString;							# Converts the parsed area to XML in a buffer

  print $XML_output_file $XML_output_text;							# Write the XML in a buffer to the the output XML file

  $XML_chunk -> dispose;                                                                        # clean up memory
}

# Process this table
#
# Save the job scheduling parameters (MAXWAIT, DAYS, etc) in all jobs, and remove these parameters.  These paramerers will be used when adding
# an RBC/TAG to the table.
# However, if the RBC/TAG or MINIMUM parameter is present in the job, don't save the job scheduling parameters, and do not remove the parameters
sub Process_the_table
{ 
  
  # Initialize all the arrays and thir pointers
  @jobArray               = "";
  @RBCarray               = "";
  @Unique_RBCs_TAGs_list  = "";    
  $JA_p    	          = 0;     	 
  $RBC_p    	          = 0;     	  	 
  $Unique_RBCs_TAGs_p     = 0;     	 


  # If this is a 700 or higher Smart table, check if the TAG parameter is being used 
  if ($Parm_Table_type eq $SMART_table)   # A TAG only appears in a Smart table
  {
    foreach my $RBC_TAG ($XML_chunk->getElementsByTagName($V630_TAG_in_table_element_name))
    {
      my $RBC_attribute  = Get_RBC_attribute_value("TAG_NAME",$RBC_TAG);
      if (There_is_a_value($RBC_attribute))
      {
        $Parm_RBC_TAG_in_table_element_name   = $V630_TAG_in_table_element_name;
	$Parm_RBC_TAG_in_job_element_name     = $V630_TAG_in_job_element_name;
	$Parm_RBC_TAG_name_in_job_element     = $V630_TAG_name_in_job_element; 
      }

    }    
  }



  # Save the table and group name for possible use in generating the RBC/TAG name, and for setting the MEMNAME/JOBNAME and GROUP parameters
  # in the table
  foreach my $smartTable ($XML_chunk->getElementsByTagName($Parm_start_table_element))
   { 
     my $Curr_parameter     = "TABLE_NAME";
     $Saved_table_name      = $smartTable->getAttribute($Curr_parameter);
     $Curr_parameter 	    = "GROUP";
     $Saved_group_name      = $smartTable->getAttribute($Curr_parameter);
     if (!There_is_a_value($Saved_group_name))
     {
       $Saved_group_name = "";
     }
     
     Display_progress_bar();
     
     if (SUB_TABLE_present())   # If the SUB_TABLE parameter is present, a warning message is issued, and the table is not modified
     {
       $No_tables_with_SUB_TABLE++;
       return;
     }
   } 


  # Save the job parameters that will be moved to an RBC.
  # In addition, if definition active from/until dates are present, set RULE_BASED_CALENDAR_RELATIONSHIP/TAG_RELATIONSHIP="AND" and WEEKDAYS="ALL"  
  foreach my $schedulingParameter ($XML_chunk->getElementsByTagName($JOB_element))
  {
     $JA_p++;   		# Update pointer to the job array

     $No_jobs_processed++;   		

     Save_job_parameters_for_moving_to_an_RBC($JA_p,$schedulingParameter);
  }
 
 

  # If this is a regular table:
  #   1. Add some attributes such as MEMNAME/JOBNAME, GROUP, etc
  #   2  If a non-default value was specified, set the attributes for the ADJUST_COND parameter in the table 
  #   3. Change "TABLE" to "SMART_TABLE" (700) or SCHED_TABLE to SCHED_GROUP (630)
  if ($Parm_Table_type eq $Regular_table)
  {
    Add_table_attributes();   # Add attributes ADJUST_COND, MEMNAME/JOBNAME, GROUP, etc 
    Change_to_SMART_Group();  # Change "TABLE" to "SMART_TABLE" (700) or SCHED_TABLE to SCHED_GROUP (630)
	
    foreach my $tableParameter ($XML_chunk->getElementsByTagName($Parm_start_table_element))
    {
       $tableParameter->setTagName($Parm_non_regular_table_changed_element);
       
       if ($Parm_start_table_element ne $Parm_non_regular_table_changed_element)
       {
	  Display_on_changed_list_file('YNY',"Changed $Parm_start_table_element to $Parm_non_regular_table_changed_element "); # Display the table, not the job name, newline before display
       }
    }
    $Parm_start_table_element = $Parm_non_regular_table_changed_element;    # This table is now a SMART/Group table
  }

  
  
  #Save the scheduling parameters, and the RBC/TAG name, of all the RBCs/TAGs in an array  	
  foreach my $RBC_TAG ($XML_chunk->getElementsByTagName($Parm_RBC_TAG_in_table_element_name))
  {
     $RBC_p++;   		# Update pointer to the RBC/TAG array

     Save_RBC_TAG_parameters($RBC_p,$RBC_TAG);
  }


  # Determine the name of the RBCs/TAG for all the entries in the job array  	
  for (my $i = 1; $i <= $JA_p; $i++)
  {
    if ($jobArray [$i] ne '')
    {
      my $RBC_TAG_name = Determine_the_RBC_TAG_name_for_current_entry($i,$JA_p);
      $jobArray [$i] [$JA_array_RBC_TAG_name] = $RBC_TAG_name;
    }	         		
  }
  
  
  # Create the RBCs/TAGs in the table
  foreach my $smartTable ($XML_chunk->getElementsByTagName($Parm_start_table_element))     # Get the SMART/Group table element
  {
    my $firstJob = $smartTable->getElementsByTagName($JOB_element)->item(0);		   # Get the first job
    
    $Unique_RBCs_TAGs_p    	 = 0;       # Pointer to the unique RBCs/TAGs array
    @Unique_RBCs_TAGs_list       = '';      # List of unique RBCs/TAGs
    
    $JA_p = 0; 
    
    foreach my $nextJob ($smartTable->getElementsByTagName($JOB_element))                  # Get the next job
    {
      $JA_p++;
      if ($jobArray [$JA_p] ne '')       # Does an table RBC/TAG have to be created for this job?  
      {
        Add_an_RBC_in_the_table_and_populate($smartTable,$firstJob,$JA_p);
      }        
    }
  } 


  # Add an RBC/TAG after each job in the table
  foreach my $smartTable ($XML_chunk->getElementsByTagName($Parm_start_table_element))     # Get the SMART/Group table element
  {
    $JA_p = 0; 
    
    foreach my $nextJob ($smartTable->getElementsByTagName($JOB_element))                  # Get the next job
    {
      $JA_p++;
      Display_job_being_processed($nextJob);
      if ($jobArray [$JA_p] ne '')                                                         # Does an table RBC/TAG have to be added after this job?  
      {
        Add_an_RBC_TAG_after_this_job($smartTable,$nextJob,$JA_p);                         # Add an RBC/TAG after this job
      }        
    }
  }
}






# Change a regular table: 
#   1. Add some attributes such as MEMNAME/JOBNAME, GROUP, etc
#   2  If a non-default value was specified, set the attributes for the ADJUST_COND parameter in the table 
sub Add_table_attributes
{  
  print $changeList_file "\n";
      
  foreach my $smartTable ($XML_chunk->getElementsByTagName($Parm_start_table_element))
  {      
    if ($Adjust_condition ne 'N')
    {
      $smartTable -> removeAttribute($Adjust_condition); 			            # Remove it from XML
      Set_table_attribute($smartTable,'ADJUST_COND',$Adjust_condition);
    }
    Set_table_attribute($smartTable,$Job_memname_jobname,$Saved_table_name);
    Set_table_attribute($smartTable,'GROUP',$Saved_table_name);
    Set_table_attribute($smartTable,$MAXWAIT,$Default_maxwait);
    Set_table_attribute($smartTable,'OWNER',$Owner);
    Set_table_attribute($smartTable,'PRIORITY',$Default_priority);
    Set_table_attribute($smartTable,'APPLICATION',$Default_application);
  }     
}


# Set the attribute for the SMART/group table 
sub Set_table_attribute
{
  my($smartTable,$Curr_parameter,$Value) = @_;
  
  $smartTable -> setAttribute($Curr_parameter,$Value);
  Display_on_changed_list_file('YNN',"Added attribute to SMART/Group table. $Curr_parameter=$Value"); # Display the table, not the job name, , no newline before display

}



# Change "TABLE" to "SMART_TABLE" (700) or SCHED_TABLE to SCHED_GROUP (630)
sub Change_to_SMART_Group
{
  my($Curr_parameter,$Value) = @_;
  
  if ($Parm_Table_type eq $Regular_table)
  {
    foreach my $tableParameter ($XML_chunk->getElementsByTagName($Parm_start_table_element))
    {
       $tableParameter->setTagName($Parm_non_regular_table_changed_element);
       if ($Parm_start_table_element ne $Parm_non_regular_table_changed_element)
       {
	  Display_on_changed_list_file('YNY',"Changed $Parm_start_table_element to $Parm_non_regular_table_changed_element");        # Display the table, not the job name, , newline before display
       }
    }
    $Parm_start_table_element = $Parm_non_regular_table_changed_element;    # This table is now a SMART/Group table
  }

  
}


# Save the parameters of this job.  These parameters will be used when adding an RBC/TAG to the table
# In addition, if definition active from/until dates are present, set RULE_BASED_CALENDAR_RELATIONSHIP/TAG_RELATIONSHIP="AND" and WEEKDAYS="ALL"  
sub Save_job_parameters_for_moving_to_an_RBC
{
  my($jobP,$schedulingParameter) = @_;      

  # Determine if this is an
  # EM   XML(parameter JOBNAME is present in a job)
  #                      or
  # z/OS XML (parameter MEMNAME is present in a job)
  
  my $One_value_is_present = $FALSE; 				    # Assume that there are no values to be saved in an RBC/Tag  
  $XML_is_EM = $TRUE;  # Assume that this is an EM XML
  $Saved_memname_jobname = Get_job_attribute_value($EM_job_JOBNAME,$schedulingParameter);  
  $Job_memname_jobname = $EM_job_JOBNAME;
  
  if (!There_is_a_value($Saved_memname_jobname))   # Is this an EM XML?
  {
    $XML_is_EM = $FALSE;
    $Job_memname_jobname = $zOS_job_MEMNAME;
    $Saved_memname_jobname = Get_job_attribute_value($zOS_job_MEMNAME,$schedulingParameter);
  }

  # Do not save the parameters of this job if MINIMUM is present, or the job has an RBC/TAG and a tag relationship of "AND",
  # or there are no scheduling parameters to be saved
  if (MINIMUM_parameter_in_job($schedulingParameter) | RBC_TAG_parameter_in_job_and_AND_relationship($schedulingParameter))   
  {                                                              	    
    $jobArray [$jobP] = '';   					           # There are no parameters that are to be moved to an RBC/TAG for this job
  }
  else
  {
    print $changeList_file "\n";
    
    for (my $i = 0; $i <= $Parm_no_scheduling_parameters; $i++)
    {
      my $List_parameter = $Parm_job_scheduling_parameters_list [$i];

      my $Job_attribute  = Get_job_attribute_value($List_parameter,$schedulingParameter);
      
      if (There_is_a_value($Job_attribute))
      {
	$jobArray [$jobP] [$JA_scheduling_parms] [$i] = $Job_attribute;     # Save the attribute value in the array
	Remove_an_attribute($schedulingParameter,$List_parameter);          # Remove the attribute from the job (but not MAXWAIT/RETRO)
	$One_value_is_present = $TRUE; 
      }
      else
      {
	$jobArray [$jobP] [$JA_scheduling_parms] [$i]  = $Empty_value;      # Indicate that no parameter value is present
      }
    }
    
      if (!$One_value_is_present)
      {
	$jobArray [$jobP] = '';                  		           # There are no parameters that are to be moved to an RBC/TAG for this job
      }
      else
      {
	Maybe_set_job_relationship_and_weekdays($schedulingParameter);
      }	
  }
}


# If the definition active from/until dates are present, set RULE_BASED_CALENDAR_RELATIONSHIP/TAG_RELATIONSHIP="AND" and WEEKDAYS="ALL"
# and set all the months to "1"
# If this is not done, and the job is uploaded to CONTROL-M for z/OS, the from/until dates are ignored
sub Maybe_set_job_relationship_and_weekdays
{
  my($schedulingParameter) = @_;      

  $Saved_definition_active_from = '';
  $Saved_definition_active_until = '';
  
  my $Job_attribute  = Get_job_attribute_value($Definition_active_from,$schedulingParameter);

  if (There_is_a_value($Job_attribute))
  {
    $Saved_definition_active_from = $Job_attribute;  
  }

  $Job_attribute  = Get_job_attribute_value($Definition_active_until,$schedulingParameter);

  if (There_is_a_value($Job_attribute))
  {
    $Saved_definition_active_until = $Job_attribute;  
  }
  
  if (($Saved_definition_active_from ne '') | ($Saved_definition_active_until ne ''))
  {
    $schedulingParameter -> removeAttribute($V630_TAG_relationship_job_element); 	         # Remove TAG_RELATIONSHIP from XML
    $schedulingParameter -> removeAttribute($V700_RBC_relationship_job_element); 	         # Remove RULE_BASED_CALENDAR_RELATIONSHIP from XML
    Add_months_to_job($schedulingParameter);                                                                         # Add the months to the job
    $schedulingParameter -> setAttribute($Parm_RBC_TAG_relationship_job_element,'AND');
    Display_on_changed_list_file('YYN',"Added to job: $Parm_RBC_TAG_relationship_job_element=AND");  # Display the table and job names, no newline before display
    $schedulingParameter -> setAttribute($Weekdays,'ALL');
    Display_on_changed_list_file('YYN',"Added to job: $Weekdays=ALL");                           
  }
}






# Save the scheduling parameters of this RBC/TAG, and the name of the RBC/TAG 
sub Save_RBC_TAG_parameters
{
  my($RBC_p,$RBC_TAG_parameter) = @_;
  my $RBC_attribute  = '';
  for (my $i = 0; $i <= $Parm_no_scheduling_parameters; $i++)
  {
    my $List_parameter = $Parm_job_scheduling_parameters_list [$i];
    $RBC_attribute  = Get_RBC_attribute_value($List_parameter,$RBC_TAG_parameter);          
    if (There_is_a_value($RBC_attribute))
    {
      $RBCarray [$RBC_p] [$RBC_scheduling_parms] [$i]  = $RBC_attribute;                 # Save the attribute value in the array
    }
    else
    {
      $RBCarray [$RBC_p] [$RBC_scheduling_parms] [$i]  = $Empty_value;                  # Indicate that no parameter value is present
    }
  }
  $RBC_attribute  = Get_RBC_attribute_value($Parm_RBC_TAG_name_in_job_element,$RBC_TAG_parameter);
  $RBCarray [$RBC_p] [$RBC_array_TAG_name] = $RBC_attribute;                            # Save the RBC/TAG name in the array
}



# Add an RBC/TAG to the table, and populate it with the attributes
sub Add_an_RBC_in_the_table_and_populate
{
  my($smartTable,$firstJob,$jobP) = @_;
  
  # Set the attributes of this RBC/TAG from the job saved parameters 
  my $RBC_name = $jobArray [$jobP] [$JA_array_RBC_TAG_name];
  
  my $Add_RBC = $TRUE;  # Assume that this RBC/TAG should be added
  
  if ($jobArray [$jobP] eq '')                             # Does job have any scheduling parameters?
  {
    $Add_RBC = $FALSE;
  }
  
  if (Parm_on_list($RBC_name,@Unique_RBCs_TAGs_list))     # This RBC/TAG was already added to the table?  
  {
    $Add_RBC = $FALSE;    
  }

  # If this RBC/TAG already exists in the table, don't add it
  if (RBC_TAG_already_in_table($RBC_name))
  {
    $Add_RBC = $FALSE;
  }

  
  # If this job doesn't have scheduling parameters
  #                 or
  # this RBC/TAG was already added to the table
  #                 or
  # this RBC/TAG already exists on the table
  #     Do  not add this RBC/TAG
  if ($Add_RBC)     			
  {
    $Unique_RBCs_TAGs_p++;
    $Unique_RBCs_TAGs_list [$Unique_RBCs_TAGs_p] = $RBC_name;
    
    my $RBC_element = $XML_chunk->createElement($Parm_RBC_TAG_in_table_element_name);    # Create the element node 
    my $Inserted = $smartTable->insertBefore($RBC_element,$firstJob);                    # Insert this element node before the first job
    $Inserted  -> setAttribute($Parm_RBC_TAG_name_in_job_element,$RBC_name);             # Add this attribue and value in this RBC/TAG
    Display_on_changed_list_file('YNY',"Inserted an RBC/Tag in table. NAME=$RBC_name");   # Display the table, not job name, newline before display
    
    for (my $i = 0; $i <= $Parm_no_scheduling_parameters; $i++)
    {
      my $Curr_parameter 	= $Parm_job_scheduling_parameters_list [$i];
      my $Curr_value 		= $jobArray [$jobP] [$JA_scheduling_parms] [$i];
      if ($Curr_parameter eq $RETRO)  # Do not add RETRO to the RBC/TAG (not supported)
      {
	$Curr_value = $Empty_value;
      }
      
      if ($Curr_value ne $Empty_value)     			                                  # Is the value of the attribute empty?
      {
	$Inserted  -> setAttribute($Curr_parameter,$Curr_value);                                  # Add this attribue and value to this this RBC/TAG
	Display_on_changed_list_file('YN',"Inserted RBC/Tag parameter in table. $Curr_parameter=$Curr_value");     # Display the table, not job name
      }
    }      
  }

}  


# Add an RBC/TAG after this job
sub Add_an_RBC_TAG_after_this_job    
{
  my($smartTable,$nextJob,$jobP) = @_;	  
    
  my $RBC_name      = $jobArray [$jobP] [$JA_array_RBC_TAG_name];                        # The RBC/TAG name
 
  if (!There_is_a_value($RBC_name))
  {
    return;
  }

  $Saved_memname_jobname = Get_job_attribute_value($Job_memname_jobname,$nextJob);
  my $RBC_TAG_element = $XML_chunk->createElement($Parm_RBC_TAG_in_job_element_name);         # Create the element node
  $RBC_TAG_element->setAttribute($Parm_RBC_TAG_name_in_job_element,$RBC_name);                # Add this attribue and value in this RBC/TAG
  Display_on_changed_list_file('YYY',"Inserted an RBC/TAG after this job.  NAME=$RBC_name");  # Display the table and job names, newline before display
  $nextJob->appendChild ($RBC_TAG_element);                                                   # Add this RBC/TAG after the job
}



# Check if there is a MINIMUM parameter in the job.  If present, display a warning message 
sub MINIMUM_parameter_in_job
{
  my($schedulingParameter) = @_;
  
  $return = $FALSE;				             # Assume that the MINIMUM parameter is not on the job
  
  my $MINIMUM = Get_job_attribute_value('MINIMUM',$schedulingParameter);          # Get the minimum job parameter 
   
  if (There_is_a_value($MINIMUM))                                          
  { 
    my $Temp_memname_jobname = Get_job_attribute_value($Job_memname_jobname,$schedulingParameter);
    Issue_warning_message('job '.$Temp_memname_jobname." contains a MINIMUM/PDS parameter.  An RBC/TAG will not be added to this job.\n\n");
    $return = $TRUE;
  }
  return $return;   
}

# If the job has an RBC/TAG parameter and TAG_RELATIONSHIP="AND", display an error message, and do not modify this job 
sub RBC_TAG_parameter_in_job_and_AND_relationship
{
  my($schedulingParameter) = @_;
  
  $return = $FALSE;     						  # Assume that the RBC/TAG with "AND" is not present in the job
 
  $RBC_TAG = '';
  
  foreach $RBC_TAG ($schedulingParameter->getElementsByTagName($Parm_RBC_TAG_in_job_element_name))   
  {
   $No_jobs_with_RBC_TAG++;
   my $RBC_TAG_relationship_attribute  = Get_RBC_attribute_value($Parm_RBC_TAG_relationship_job_element,$schedulingParameter);
   if ($RBC_TAG_relationship_attribute eq "AND")
   {
       $return = $TRUE;
       my $Temp_memname_jobname = Get_job_attribute_value($Job_memname_jobname,$schedulingParameter);
       Issue_warning_message('job '.$Temp_memname_jobname." contains a TAG/RBC relationship of 'AND'.  An RBC/TAG will not be added to this job.\n\n");
   }
   last;     # Only need to know if there is one RBC/Tag in the job
  }
  
  return $return;  
}

# Determine the TBC/TAG name for the current entry in the job array
sub Determine_the_RBC_TAG_name_for_current_entry
{
  my($Current_job_array_pointer,$Maximum_entries_in_job_array) = @_;             
  
  my $RBC_TAG_name = '';
  my $Temp         = '';
  
  # Check if this entry already contains an RBC/TAG name.  If yes, return the name
  $RBC_TAG_name = $jobArray [$Current_job_array_pointer] [$JA_array_RBC_TAG_name];
  if (There_is_a_value($RBC_TAG_name))    # an RBC/TAG value exists?
  {
    return $RBC_TAG_name;
  }
  
  #Check if the job parameters match an existing RBC.  If yes, return  this name 
  $RBC_TAG_name = Job_parms_match_RBC_parms($Current_job_array_pointer);
  if ($RBC_TAG_name ne '')    # if there was a match on the parameters, return the RBC/TAG name
  {
    return $RBC_TAG_name;
  }
  		
  # Determine the RBC/TAG name based on the scheduling parameters of all the jobs in the table.
  #
  # Assume that there are three jobs with the following sample schedule data:
  #
  # Job # DAYS    WDAYS   DCAL
  #  1    10      20      DCAL1
  #  2    10      20      DCAL2
  #  3    10      20      DCAL1
  #
  # Since all the jobs have the same DAYS and WDAYS parameters, these two parameters will not be involved in generating the RBC/TAG name.
  # Two unique RBC/TAG names will be generated:
  #   A. DCAL_DCAL1
  #   B. DCAL DCAL2
  for (my $i = 1; $i <= $Maximum_entries_in_job_array; $i++)
  {
    if ($i ne $Current_job_array_pointer)   			# Ignore the current entry
    {
      for (my $s_p = 0; $s_p <= $Parm_no_scheduling_parameters; $s_p++)
      {
        if ($jobArray [$i] [$JA_scheduling_parms] [$s_p] ne $jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p])
	{
	  my $Temp_name = $Parm_job_scheduling_parameters_list[$s_p].'_'.$jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p].'_';  
	  my $Temp_len = length($Temp_name);
	  my $Temp_RBC_TAG = substr($RBC_TAG_name,0,$Temp_len);
	  if (substr($Temp_name, 0, $Temp_len) ne $Temp_RBC_TAG)    # Make sure that duplicate names are not generated
	  {
	    $RBC_TAG_name = $RBC_TAG_name.$Parm_job_scheduling_parameters_list[$s_p].'_'.$jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p].'_';  
	  }
	}
      }
    }  
  }        		
  chop $RBC_TAG_name;   # chop the extra underline character at the end of the string
  
  # If the length of the RBC/TAG is greater then the maximum allowed RBC/TAG length, or there is no RBC, set the RBC/TAG name
  if ((length($RBC_TAG_name) > $Parm_max_RBC_TAG_length) | (!There_is_a_value($RBC_TAG_name)))   
  {
    $RBC_TAG_name = Set_the_RBC_TAG_name();     # Set the RBC/TAG name based on user-supplied prefix  
  }
  else
  {
      # Remove invalid characters such as commas because on the M/F, RBC/Tag can contain commas etc -- but not in EM.  
      $RBC_TAG_name = Remove_invalid_characters_from_RBC_TAG_name($RBC_TAG_name);     
  }
   
  # All jobs with the same scheduling parameters will be set with the same RBC/TAG name
  Set_RBC_TAG_for_all_matching_jobs($RBC_TAG_name,$Current_job_array_pointer,$Maximum_entries_in_job_array);         

  return $RBC_TAG_name;    
} 




# All jobs with the same scheduling parameters will be set with the same RBC/TAG name
sub Set_RBC_TAG_for_all_matching_jobs
{
  my ($RBC_TAG_name,$Current_job_array_pointer,$Maximum_entries_in_job_array) = @_;
  
  for (my $i = 1; $i <= $Maximum_entries_in_job_array; $i++)
  {
    if ($i ne $Current_job_array_pointer)   			# Ignore the current entry
    {
      my $match = $TRUE;  # Assume that all jb parameters match
      for (my $s_p = 0; $s_p <= $Parm_no_scheduling_parameters; $s_p++)
      {
        if ($jobArray [$i] [$JA_scheduling_parms] [$s_p] ne $jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p])
	{
	  $match = $FALSE;   
	}
      }
      if ($match)
      {
	$jobArray [$i] [$JA_array_RBC_TAG_name] = $RBC_TAG_name;
      }
    }  
  }        	
}
 
 
 
 
# Set the RBC/TAG name based on the group name or table name or a user-supplied prefix or a default prefix
sub Set_the_RBC_TAG_name
{
  my $Generated_RBC_TAG_prefix = $Parm_RBC_TAG_prefix;
  
  if ($Parm_RBC_TAG_prefix eq $RBC_TAG_group_prefix)
  {
    if ($Saved_group_name ne '')
    {
      $Generated_RBC_TAG_prefix = $Saved_group_name;
    }
    else
    {
      $Generated_RBC_TAG_prefix = $Parm_RBC_TAG_no_group_prefix; 
    }
  }
  elsif ($Parm_RBC_TAG_prefix eq $RBC_TAG_table_prefix)
  {

    $Generated_RBC_TAG_prefix = $Saved_table_name;

  }

  $RBC_TAG_name_seq++;		# The sequence number for the generated RBCs/TAGs

  return $Generated_RBC_TAG_prefix.'_'.$RBC_TAG_name_seq;
}





# Check if the job parameters match an existing RBC, and if yes return the RBC/TAG name
sub Job_parms_match_RBC_parms
{
  my ($Current_job_array_pointer) = @_;  
  
  # A regular table doesn't have any RBCs
  if ($Parm_Table_type eq $Regular_table)
  {
    return '';
  }
  
  for (my $RBC_curr_p = 1; $RBC_curr_p <= $RBC_p; $RBC_curr_p++)
  {
    $this_RBC_matches = $TRUE;   # assume that the parameters of this RBC/TAG matches the job parameters
    for (my $s_p = 0; $s_p <= $Parm_no_scheduling_parameters; $s_p++)
    {
      if ($jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p] ne $RBCarray [$RBC_curr_p] [$RBC_scheduling_parms] [$s_p])
      {
	$this_RBC_matches = $FALSE;
      }
    }
    if ($this_RBC_matches)
    {
      return $RBCarray [$RBC_curr_p] [$RBC_array_TAG_name];   # return the RBC/TAG name
    }
  }
  return '';
} 
  
 

# Build the RBC/TAG name
sub Build_RBC_TAG_name 	
{
  my($i,$Current_job_array_pointer) = @_;            
 
  my  $RBC_TAG_name = '';
  
  for (my $s_p = 0; $s_p <= $Parm_no_scheduling_parameters; $s_p++)
  {
    if ($jobArray [$i] [$JA_scheduling_parms] [$s_p] ne $jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p])
    {
      $RBC_TAG_name = $RBC_TAG_name.$Parm_job_scheduling_parameters_list[$s_p].'_'.$jobArray [$Current_job_array_pointer] [$JA_scheduling_parms] [$s_p].'_';
    }  
  }      
  return $RBC_TAG_name; 
   
}

  

# Is the current RBC/TAG have the same scheduling parameters as for the current job?
sub Match_current_RBC_TAG 	
{
  # The pointer to the current RBC/TAG array, the pointer to the job array, and the number of scheduling parameters  
  my($RBC_TAG_p,$jobP) = @_;            
  
  $Current_RBC_TAG_sched_parm_array = '';

  if (defined($RBC_TAG_Array [$RBC_TAG_p]))
  {
    $Current_RBC_TAG_sched_parm_array = $RBC_TAG_Array [$RBC_TAG_p] [$RBC_scheduling_parms];   # An array of scheduling parameters
    
    for (my $j = 0; $j <= $Parm_no_scheduling_parameters; $j++)
    {
      if ($Current_RBC_TAG_sched_parm_array [$j] ne $jobArray [$jobP] [$JA_scheduling_parms] [$j])
      {
	return $FALSE;    # No match   
      }
    }
  }
  else
  {
    return $FALSE;
  }
  
  return $TRUE;
    
}



# Get a job attribute value 
sub Get_job_attribute_value
{
  my ($JobParm,$schedulingParameter) = @_;

  my $return = ''; 			                   # Assume that the job parameter is not found

  $Temp = $schedulingParameter->getAttribute($JobParm);
  
  if (length($Temp) > 0)    			   	   # Is this parameter in the job?                                 
  {
    $return = $Temp; 					   # Return the job parameter
  }

  return $return;
  
}



# Get a RBC attribute value 
sub Get_RBC_attribute_value
{
  my ($RBCParm,$RBCParameter) = @_;

  my $return = ''; 			                   # Assume that the RBC/TAG parameter is not found

  $Temp = $RBCParameter->getAttribute($RBCParm);
  
  if (length($Temp) > 0)    			   	   # Is this parameter in the RBC?                                 
  {
    $return = $Temp; 					   # Return the RBC/TAG parameter
  }

  return $return;
  
}


# Check if this parameter is on the list
sub Parm_on_list
{
  my($parameter,@list) = @_;

  my $return = $FALSE;              # Assume that the parm is not on the list  
  
  if ($parameter ~~ @list)          # Is the parameter on the list?
  {
    $return = $TRUE;               # Parm is on the list
  }
  
  return $return; 
}  



# Check if the parameter has a value
sub There_is_a_value
{
  my($parameter) = @_;

  my $Temp = join(' ', split(' ', $parameter));  # Remove leading and trailing blanks
  
  my $return = $FALSE;            		 # Assume that the parameter doesn't have a value  
  
  if (length($Temp) > 0)       	                 # Does the parameter have a value?
  {
    $return = $TRUE;                             # Parameter has a value
  }
  
  return $return; 
} 

sub Issue_warning_message
{
  my($Message) = @_;            # The warning message to be printed
  print $Exceptions_file "$Message";
}



# Display a change message on the change list file
sub Display_on_changed_list_file
{
 my($whatTodisplay,$Message_to_display) = @_;  # $whatTodisplay - display table Y/N, display job Y/N, display newline Y/N 
 
 my $temp_msg = '';
 
  if (substr($whatTodisplay,2,1) eq 'Y')   # Newline?
 {
   $temp_msg = $temp_msg."\n";
 }
 
 if (substr($whatTodisplay,0,1) eq 'Y')   # Display the table name?
 {
   $temp_msg = $temp_msg.'table='.$Saved_table_name.' ';
 }
 
 if (substr($whatTodisplay,1,1) eq 'Y')   # Display the memname (z/OS) / job name (EM)?
 {
   $temp_msg = $temp_msg.'job='.$Saved_memname_jobname.' ';
 }
 
 print $changeList_file $temp_msg.$Message_to_display."\n";
 
}




sub Display_progress_bar
{
    
    $Current_progress_count++;
    $Curr_table_number_progress++;
	
    my $Msg = "Processing table: ".$Saved_table_name.". Time: ".scalar(localtime(time())).". Table number ".$Curr_table_number_progress." of ".$Total_tables_progress.". Number of records in table: ".$Parm_number_of_records_in_table."\n\n";
    Print_progress_msg($Msg);
    if ($Current_progress_count eq $Display_progress_after_this_count)
    {
      print $progress->update($Curr_table_number_progress)->string()."\r";   
      $Current_progress_count = 0;
    }
}




sub Display_job_being_processed
{
  my($nextJob) = @_;
  my $jobName = Get_job_attribute_value($Job_memname_jobname,$nextJob);
  
  $Parm_job_number++;
  my $Msg = "Processing job: ".$jobName.". Job number: ".$Parm_job_number." in table: ".$Saved_table_name.". Time: ".scalar(localtime(time()))."\n\n";
  Print_progress_msg($Msg);
}




# Check if this RBC/TAG is already in the table
sub RBC_TAG_already_in_table
{
  my($tagName) = @_;

  for (my $RBC_curr_p = 1; $RBC_curr_p <= $RBC_p; $RBC_curr_p++)
  {
    if ($tagName eq $RBCarray [$RBC_curr_p] [$RBC_array_TAG_name])
    {
      return $TRUE;
    } 
  }
  
  return $FALSE;
  
}





sub Print_progress_msg
{
  my($Msg) = @_;
  print $Progress_file $Msg;
  #print $Msg;  
}

# If the SUB_TABLE parameter is present, issue warning message, and don't touch this table
sub SUB_TABLE_present
{

 my $return	   = $FALSE;    # Assume that a SUB_TABLE is not present
 my $Saved_jobname = '';
 
 if ($Parm_Table_type eq $SMART_table)   # A SUB_TABLE only appears in a Smart table
 {
    foreach my $schedulingParameter ($XML_chunk->getElementsByTagName($SUB_TABLE_element))
    {
       $Saved_jobname = Get_job_attribute_value($EM_job_JOBNAME,$schedulingParameter);   # Look for JOBNAME within the SUB_TABLE
       if (There_is_a_value($Saved_jobname))   # Is JOBNAME present within the SUB_TABLE?
       {
         $return	   = $TRUE;   # SUB_TABLE is present
	 print $Exceptions_file "Table ".$Saved_table_name." contains a non-supported SUB_TABLE element.  This table was not modified\n";
       }
       last;
    }
 }

 return $return;
 
}



# Remove the attribute from the job (not MAXWAIT/RETRO)
sub Remove_an_attribute
{
  my($schedulingParameter,$List_parameter) = @_;  

  if ($List_parameter eq $RETRO)   # Do not remove the RETRO attribute from the job (not supported in the RBC/TAG)
  {
    return;
  }
  
  if ($List_parameter ne $MAXWAIT)   # Do not remove the MAXWAIT attribute from the job, but do copy it to the RBC/Tag
  {
    $schedulingParameter->removeAttribute($List_parameter);             # Remove it from XML
    # Display the table and job names, no newline before display
    Display_on_changed_list_file('YYN',"Removed attribute $List_parameter from the job, will be moved to an RBC/Tag");
  }
  else
  {
    Display_on_changed_list_file('YYN',"Attribute $List_parameter will be moved to an RBC/Tag");
  }
}


sub niceSize
{
	# Will work up to considerable file sizes!
	my $fs = $_[0];	# First variable is the size in bytes
	my $dp = $_[1];	# Number of decimal places required
	my @units = ('bytes','KB','MB','GB','TB','PB','EB','ZB','YB');
	my $u = 0;
	$dp = ($dp > 0) ? 10**$dp : 1;
	while($fs > 1024)
	{
	  $fs /= 1024;
	  $u++;
	}
	if($units[$u]){ return (int($fs*$dp)/$dp)." ".$units[$u]; } else{ return int($fs); }
}


# On M/F, RBC/Tag can contain commas etc -- but not in EM.  Therefore, remove invalid characters 
sub Remove_invalid_characters_from_RBC_TAG_name     
{
   my($RBC_TAG_name) = @_;

   $RBC_TAG_name =~ s/,/_/g;     # Change commas to underlines so that a RBC/Tag name like "WEEKDAYS_6,5" is translated to "WEEKDAYS_6_5" 
   $RBC_TAG_name =~ s/(\W)//g;   # Apphanumeric and the underline character are considred valid.  Any invalid characters are removed.  
   
   return $RBC_TAG_name;
}



# Print the release of the XML (630/700) -- only once
sub Display_XML_release
{
  my($Parm_release) = @_;

  if (length($Release) ne 0)   # print the release only once
  {
    return;  
  }
  
  $Release = $Parm_release;
  
  my $Release_msg = 'Processing '.$Parm_release." XML file\n\n";   
  
  print $Exceptions_file $Release_msg;

  print $changeList_file $Release_msg;

  print $Progress_file $Release_msg;
  
  print $Release_msg;
}


# Change o/p XML file name -- add "OUT" to name of file
# If input file is UTIL.XML, output file is UTIL.OUT.XML 
sub Change_XML_output_file_name
{
  my($Parm_XML_Output_file_name) = @_;

  $Parm_XML_Output_file_name = uc($Parm_XML_Output_file_name);
  
  if ($Parm_XML_Output_file_name =~ m/.XML/)
  {
    $Parm_XML_Output_file_name =~ s/.XML/.OUT.XML/;
  }
  else
  {
    $Parm_XML_Output_file_name = $Parm_XML_Output_file_name . ".OUT"  
  }

return $Parm_XML_Output_file_name;

}


# Add the months to the job
sub Add_months_to_job
{
    my($schedulingParameter) = @_;

    my $No_of_months = (scalar @Months_list) - 1;
    
    for (my $i = 0; $i <= $No_of_months; $i++)
    {
      $schedulingParameter -> setAttribute($Months_list[$i],'1');
    }

return;

}







sub Debug_print_XML_buffer
{
  my($function,$theNode) = @_;


  return;   # not needed for now
   
  if ($function eq 'open')
  {
    unless(open($Debug_Print_File, '> $XML_Debug_Print'))
    {
      die('Could not create ', $XML_Debug_Print, " file. Aborting..\n");
    }
    return;
  }
 
   if ($function eq 'close')
  {
    close ($Debug_Print_File);
    return;
  }
  
  my $thisType = $theNode -> getNodeType;
  my $XML_chunkList = $theNode -> getChildNodes;
  my $name = $theNode -> getNodeName;
  my $attLength;
  my $length = $XML_chunkList -> getLength;

  my $attList = $theNode -> getAttributes;

  if ($attList) {
    $attLength = $attList -> getLength;
  }

  $depth = 0;		# The depth of the recursive call.
  $sep = '    ';	# The indentation separator

  print $Debug_Print_File $sep x($depth), "-NodeName: $name (type $thisType, $length children)\n";

  #Atributes are NOT part of the DOM tree, but properties of an element.#As properties, things like parent, child, next and last item are#not definable, even though it looks like a node.#If they exist, handle them.
  if ($attLength) {
    $depth++;
    for (my $j = 0; $j < $attLength; $j++) {
      my $attNode = $attList -> item($j);
      print $Debug_Print_File $sep x($depth), "*** Attribute: ", $attNode -> getName, "=\"", $attNode -> getValue, "\"\n";
    }
    $depth--;
  }

  $depth++;

  for (my $i = 0; $i < $length; $i++) {
    my $XML_temp_chunk = $XML_chunkList -> item($i);
    my $theType = $XML_temp_chunk -> getNodeType;
    my $j = $i + 1;

    if ($theType == ELEMENT_NODE) {
      print $Debug_Print_File $sep x($depth), "[$j]Element Node: ", "\<", $XML_temp_chunk -> getTagName, "\n";#Recursive call to itself
      Debug_print_XML_buffer("print",$XML_temp_chunk);
    }
    elsif($theType == TEXT_NODE) {
      my $temp = $XML_temp_chunk -> getData; #Sub out the tabs and newlines with text equivalents
      $temp =~ s/\n/\\n/g;
      $temp =~ s/\t/\\t/g;
      print $Debug_Print_File $sep x($depth), "[$j]Text Node: ", $temp, "\n";
    } else {
      die "Error in parsing XML", $sep x($depth), "[$j]UNKNOWN Node: ", $XML_temp_chunk -> getNodeName, "\n";
    }
  }

  $depth--;
}


