import xml.etree.ElementTree as ET
import csv
import argparse
import logging
from datetime import datetime
import os
import ctypes  # An included library with Python install.   

def argument_parser():
    parser = argparse.ArgumentParser(
        description="Create a CSV folder/condition xref from XML")
    parser.add_argument('-i', dest='input_file', type=str,
                        help='Input XML file', metavar='', required=True)
    parser.add_argument('-o', dest='output_file', type=str,
                        help='Output CSV file', metavar='', required=True)
    parser.add_argument('-log', dest='log', type=int,
                        help=argparse.SUPPRESS, default=0, metavar='')
    return parser

def build_folders_dict(root, folder_type, dictionary):
    for folder in root.iter(folder_type):
        folder_name = folder.attrib.get("FOLDER_NAME", "")
        inconds=[]    
        outconds=[]  
        # build a list of all inconds and outconds at job level
        for job in folder.findall("JOB"):
            for cond in job.findall("INCOND"):
                condname = cond.attrib.get("NAME")
                if condname not in inconds:
                    inconds.append(condname)
            for cond in job.findall("OUTCOND"):
                condname = cond.attrib.get("NAME")
                sign = cond.attrib.get("SIGN")
                if condname not in outconds and sign not in ('-','DEL'):
                    outconds.append(condname)
        # add all doconds (both at folder and job level - iter() method)           
        for docond in folder.iter("DOCOND"):
            condname = docond.attrib.get("NAME")
            sign = cond.attrib.get("SIGN")
            if condname not in outconds and sign not in ('-','DEL'):
                outconds.append(condname)
        # add folder level conditions                
        for cond in folder.findall("INCOND"):
            condname = cond.attrib.get("NAME")
            if condname not in inconds:
                inconds.append(condname)
        for cond in folder.findall("OUTCOND"):
            condname = cond.attrib.get("NAME")
            sign = cond.attrib.get("SIGN")
            if condname not in outconds and sign not in ('-','DEL'):
                outconds.append(condname)                
        # purge "local" inconds and mark outcond as purgeable
        # as we're removing elements, we've got to work on a copy of the list
        for incond in inconds[:]:
            if incond in outconds:
                inconds.remove(incond)
                ix = outconds.index(incond)
                outconds[ix] = '[*remove*]' + incond
        item = (inconds,outconds)
        dictionary[folder_name] = item

def remove_local_cond(folder_dictionary):
    global_incond_list  = []
    for folder_key in folder_dictionary:
        item = folder_dictionary[folder_key]
        (inconds, outconds) = item
        for incond in inconds:
            if incond not in global_incond_list:
                global_incond_list.append(incond)
    # scan all outconds and remove those marked for removal if not in global incond dictionary
    for folder_key in folder_dictionary:
        item = folder_dictionary[folder_key]
        (inconds, outconds) = item
        # we're removing elements, got to work on a copy of the list
        for outcond in outconds[:]:
            cond = str(outcond)
            if cond.startswith("[*remove*]"):
                cond = cond.removeprefix("[*remove*]")
                # if no-one is using it as an incond, we kill it
                if cond not in global_incond_list:
                    outconds.remove(outcond)
                else:
                # update name, remove "purgeable" prefix
                    ix = outconds.index(outcond)
                    outconds[ix] = cond
        # update folders dictionary            
        item = (inconds,outconds)
        folder_dictionary[folder_key] = item
        
def create_cond_dictionary(folder_dict, cond_dict):
    for folder_key in folder_dict:
        item = folder_dict[folder_key]
        (inconds, outconds) = item
        for incond in inconds:
            if incond not in cond_dict:
                cond_dict[incond] = [[folder_key],[]]
            else:
                item = cond_dict[incond]
                (inlist,outlist) = item
                if len(inlist) > 0:
                    inlist.append(folder_key)
                else:
                    inlist=[folder_key]
                item = (inlist,outlist)
                cond_dict[incond] = item
        for outcond in outconds:
            if outcond not in cond_dict:
                cond_dict[outcond] = [[],[folder_key]]
            else:
                item = cond_dict[outcond]
                (inlist,outlist) = item
                if len(outlist) > 0:
                    outlist.append(folder_key)
                else:
                    outlist=[folder_key]
                item = (inlist,outlist)
                cond_dict[outcond] = item
    
def write_xref(writer,tree,cond_dict, folder_type):

    root = tree.getroot()    
    iter=1
    total = str(len(list(root.iter(folder_type))))
    
    for folder in root.iter(folder_type):
        print("processing",folder_type.replace("_"," ").lower(),iter, "of", total, "          ", end='\r' )
        iter = iter+1
        folder_name = folder.attrib.get("FOLDER_NAME")
        local_server = folder.attrib.get("DATACENTER")
        for incond in folder.iter("INCOND"):
            condname = incond.attrib.get("NAME")
            # if condition is not in dictionary it's an "internal" one - we are not interested
            if condname in cond_dict:
                item = cond_dict[condname]
                (inlist, outlist) = item
                for remfolder in outlist:
                    remote_server = tree.find(".//"+folder_type+"[@FOLDER_NAME='"+remfolder+"']").attrib.get("DATACENTER")
                    writer.writerow([folder_name, local_server, remfolder, remote_server,"IN", condname ])
        for outcond in folder.iter("OUTCOND"):
            condname = outcond.attrib.get("NAME")
            if condname in cond_dict:
                # if condition is not in dictionary it's an "internal" one - we are not interested
                item = cond_dict[condname]
                (inlist, outlist) = item
                for remfolder in inlist:
                    remote_server = tree.find(".//"+folder_type+"[@FOLDER_NAME='"+remfolder+"']").attrib.get("DATACENTER")
                    writer.writerow([folder_name, local_server, remfolder, remote_server,"OUT", condname ])
        for docond in folder.iter("DOCOND"):
            condname = docond.attrib.get("NAME")
            # if condition is not in dictionary it's an "internal" one - we are not interested
            if condname in cond_dict:
                item = cond_dict[condname]
                (inlist, outlist) = item
                for remfolder in inlist:
                    remote_server = tree.find(".//"+folder_type+"[@FOLDER_NAME='"+remfolder+"']").attrib.get("DATACENTER")
                    writer.writerow([folder_name, local_server, remfolder, remote_server,"OUT", condname ])

def generate_log_file(log):
    dirpath = "log"
    if not os.path.isdir(dirpath):
        os.makedirs(dirpath)
    path = os.path.join(dirpath, "cond_folder_xref.log")
    if os.path.exists(path):
        os.remove(path)
    if log == 0:
        logging.basicConfig(filename=path, level=logging.INFO)
    elif log == 1:
        logging.basicConfig(filename=path, level=logging.ERROR)
    elif log == 2:
        logging.basicConfig(filename=path, level=logging.DEBUG)

def get_time_stamp():
    return str(datetime.now()) + " "

def main():
    args = argument_parser().parse_args()
    generate_log_file(args.log)
    logging.info(get_time_stamp() + ' script parameters: ' + str(args))
    
    # I'm using dictionary rather than lists in case we need to read by key
    # we do for conditions, we might for folders (processing of "both"???)
    folders={}
    conds={}
  
    print("step 0 - parsing xml file",args.input_file, end='\r' )
    logging.info(get_time_stamp() + ' parsing xml file')
    tree = ET.parse(args.input_file)
    root = tree.getroot()
    
    print("step 1 - processing folders for conditions search", end='\r' )
    
    logging.info(get_time_stamp() + ' building smart folders dictionary')    
    build_folders_dict(root, "SMART_FOLDER", folders)
    
    logging.info(get_time_stamp() + ' building "normal" folders dictionary')        
    build_folders_dict(root, "FOLDER", folders)
    
    # now we've got a dictionary of all folders, values a list of "clean" inconds and a list of outconds
    # we'll see if purgeable outconds are used anywhere else
    # build a "global" dictionary of inconds
    print("step 2 - processing conditions to build condition db", end='\r' )

    logging.info(get_time_stamp() + ' cleanse folders dictionary')        
    remove_local_cond(folders)

    # now the folders dictionary should cointain only "external" references
    # build a condition dictionary: key = cond name
    # value = tuple (list_of_folders_in, list_of_folders_out)
    logging.info(get_time_stamp() + ' create conditions dictionary')        
    conds = {}
    create_cond_dictionary(folders,conds)
    
    print("step 3 - creating csv XREF                             ", end='\r' )
    logging.info(get_time_stamp() + ' starting CSV creation')        
    done = 'N'
    while done != 'Y':
        try:
            with open(args.output_file, 'w', newline='') as csv_file:
                csv_writer = csv.writer(csv_file)
                csv_writer.writerow(['Folder Name','Control-M Server Name', 'Related Folder Name', 'Related Folder Control-M Server Name', 'Related by', 'Condition'])
                write_xref(csv_writer,tree,conds,"SMART_FOLDER")
                write_xref(csv_writer,tree,conds,"FOLDER")
                done = 'Y'
        except PermissionError:
            button = ctypes.windll.user32.MessageBoxW(0, "Please close the XREF report", "Warning. File open in Excel", 5)
            if button!=4:
                done = 'Y'
        except:
            ctypes.windll.user32.MessageBoxW(0, "Sorry, something went wrong - see output", "Error writing XREF report", 0)
            raise

    # create a condition xref if needed - 
    # with open("cond_"+args.output_file, 'w', newline='') as csv_file:                    
    #     csv_writer = csv.writer(csv_file)
    #     csv_writer.writerow(['Condition', 'Folder Name' , 'Job Name', 'Direction', 'Date/Sign'])
    #     for cond_key in conds:
    #         items = conds[cond_key]
    #         if items != None and len(items) > 0:
    #             for item in items:
    #                 (direction, folder_name, job_name, conddate) = item
    #                 csv_writer.writerow([cond_key, folder_name, job_name, direction, conddate])
    print(f"Data extracted from {args.input_file} and saved to {args.output_file}")
    logging.info(get_time_stamp() + ' all done')        

if __name__ == "__main__":
    main()
    