import json
from folder import folder
from folder_fetcher import folder_fetcher
from consts import consts


class folder_fetcher_from_json(folder_fetcher):
    #
    # This class implements the folder_fetcher interface from a CTM json file.
    #
    def __init__(self, filename):

        self._folder_ = None
        self._folders_list_ = []
        self._indx_in_list_ = 0
        self._seq_in_json_ = 0
        self._vars_sets_ = []
        self._vars_vals_ = []

        print(' ' * 80, end='\r')  # clear for spinner message...
        print('Loading input from ' + filename + '...', end='\r')  # Spinner message...

        with open(filename, mode="r", encoding="utf-8") as infile:
            json_obj = json.load(infile)

            for f in json_obj:
                fd = json_obj[f]

                if f == "Folders":
                    for fld in fd:
                        self._manage_json_folder_(fld['Name'], fld)
                else:
                    self._manage_json_folder_(f, fd)

    def _manage_json_folder_(self, item, data):
        #
        # Manage a folder in json format
        #
        if isinstance(data, dict):

            if data['Type'] != 'SubFolder':  # Starting a new parent folder (not a sub-folder)
                del self._folder_
                if 'FolderLibrary' in data:  # Add the library name to MF folders
                    folder_name = data['FolderLibrary'] + ' : ' + item
                else:
                    folder_name = item
                self._folder_ = folder(folder_name, self._seq_in_json_)
                self._seq_in_json_ = self._seq_in_json_ + 1

                self._vars_sets_ = []  # Reset the accumulation list of vars that get the values
                self._vars_vals_ = []  # Reset the accumulation list of variables used as values (if not set before they are considered globals...)

            # print("Folder name: " + item + "' Folder type: " + data['Type'])

            for subitem in data:
                subitem_data = data[subitem]
                match subitem:
                    case "Jobs":
                        for j in subitem_data:
                            self._manage_json_job_(j, j['Name'])

                    case "Variables":
                        self._manage_json_variables_(subitem_data, 'folder')

                    case "Resources":
                        self._manage_json_resources_(subitem_data, 'folder')

                    case "Folders":
                        for subf in subitem_data:
                            self._manage_json_folder_(subf['Name'], subf)

                if isinstance(subitem_data, dict) and 'Type' in subitem_data:
                    dtype = subitem_data['Type'][0:3]
                    match dtype:
                        case "Job":
                            self._manage_json_job_(subitem_data,
                                                   subitem)  # handle jobs in old json format outside the "jobs array"

                        case "Sub":
                            self._manage_json_folder_(subitem,
                                                      subitem_data)  # handle sub-folders (recursive call to self...)

                        case "Res":
                            self._manage_json_res_(subitem_data, subitem, 'folder')  # handle resources

                        case "If:" | "IfC":
                            self._manage_json_if_(subitem_data, subitem,
                                                  'folder')  # handle If blocks (may include variables and events)

                        case "Wai" | "Add" | "Del":
                            self._manage_json_event_(subitem_data, subitem, 'folder')  # handle events (wait, add, del)

            if data['Type'] != 'SubFolder':  # Add to the list only when out with the outtermost (parent) folder
                self._add_vars_not_set_as_globals_()
                self._folders_list_.append(self._folder_)

    def _manage_json_job_(self, job, job_name):
        #
        # Manage a job in json format
        # The job name is passed as a second variable because when the jobs are not within a "jobs array" the "name" is not included and is specified outside
        #
        self._folder_.add1_to_jobs_counter()
        # print ('>>>Job>>>' + job_name)

        for item in job:
            data = job[item]
            match item:
                case "Variables":
                    self._manage_json_variables_(data, 'job')  # handle variables

                case "Resources":
                    self._manage_json_resources_(data, 'job')  # handle resources

            if isinstance(data, dict) and 'Type' in data:
                dtype = data['Type'][0:3]
                match dtype:
                    case "Res":
                        self._manage_json_res_(data, item, 'job')  # handle resources

                    case "If:" | "IfC":
                        self._manage_json_if_(data, item, 'job')  # handle If blocks (may include variables and events)

                    case "Wai" | "Add" | "Del":
                        self._manage_json_event_(data, item, 'job')  # handle events (wait, add, del)

                if data['Type'] == "Action:CaptureOutput":
                    self._manage_json_capture_(data, item, 'job')  # handle capture from output (has a var in it)

    def _manage_json_variables_(self, vars, level):
        #
        # Manage variables array in json format
        #
        for v in vars:
            for var in v:
                self._manage_var_val_(var, v[var], level)

    def _manage_json_resources_(self, resrs, level):
        #
        # Manage resources array in json format
        #
        for r in resrs:
            self._manage_json_res_(r, r['Name'], level)

    def _manage_var_val_(self, var, val, level):
        #
        # Handle a single variable.
        # If its clear from the var name that this is a pool or global variable then add it to the folder resources.
        # If not - add it to the vars repository so it can later (at the end of the folder's processing) be checked if its
        # a global variable or not (i.e. variable that is referenced without a set...)
        #
        self._manage_var_name_(var, self._vars_sets_, level)

        vars_in_val = self._extract_vars_from_val_(val)
        for v in vars_in_val:
            self._manage_var_name_(v, self._vars_vals_, level)

    def _manage_var_name_(self, var, vlist, level):
        if var[0:1] == '\\':  # If starts with a "\" then its a global or pool or folder variable...
            if var[1:2] != '\\':  # If 2nd char is not "\" then this is a global variable
                self._folder_.add_resource(consts.VAR_PREFIX + var)  # ... Add it to the objects used list
            else:
                if var.find('\\',
                            2) > 0:  # if another "\" appears from the 3rd char this is a pool variable (otherwise - folder var)
                    self._folder_.add_resource(consts.VAR_PREFIX + var)  # ... pool: Add it to the objects used list
        else:
            vlist.append(var)  # Local variable. Keep it to check later if is a global (reference in val without a set)

    def _extract_vars_from_val_(self, val):
        #
        # Look for variables in the value string and return them in a list.
        #
        ret_list = []
        varsstr = val

        if isinstance(varsstr, str):
            index_percent_percent = varsstr.find('%%', 2)  # Look for %%
        else:
            return (ret_list)

        while index_percent_percent != -1:
            if varsstr[0:2] == '%%':  # if a begining of a var name - append it to the list
                ret_list.append(self._strip_var_name_(varsstr[0: index_percent_percent]))
            varsstr = varsstr[index_percent_percent:]
            if isinstance(varsstr, str):
                index_percent_percent = varsstr.find('%%', 2)
            else:
                return (ret_list)

        if varsstr[0:2] == '%%':  # if a begining of a var name - append it to the list
            ret_list.append(self._strip_var_name_(varsstr))

        return (ret_list)

    def _strip_var_name_(self, var):
        #
        # Remove the %% from var name and cut the operators and spaces at the end of the var name
        #
        ret_var = var[2:]  # Take off the %%

        for cindx, char in enumerate(
                ret_var):  # Look for the first occurance of operators and spaces and cut from there
            if char in ['+', '-', '*', '/', ' ', '.']:
                ret_var = ret_var[0: cindx]

        return (ret_var)

    def _add_vars_not_set_as_globals_(self):
        #
        # All local vars that appear in the "value" side of sets but not in the "variable" side of any other set in the folder are considered "globals"
        #
        for v in self._vars_vals_:
            if v not in self._vars_sets_:
                self._folder_.add_resource(consts.VAR_PREFIX + '\\' + v)

    def _manage_json_res_(self, res, name, level):
        #
        # Manage resources
        #
        if res['Type'] == 'Resource:Pool':
            self._folder_.add_resource(consts.QRES_PREFIX + name)
            # print(consts.QRES_PREFIX + name)
        else:
            self._folder_.add_resource(consts.RES_PREFIX + name)
            # print(consts.RES_PREFIX + name)

    def _manage_json_if_(self, ifobj, name, level):
        #
        # Manage if blocks (may include variables and events)
        #
        for i in ifobj:

            if i[0:5] == 'Event':
                self._folder_.add_resource(consts.EVENT_PREFIX + ifobj[i]['Event'])
                # print(consts.EVENT_PREFIX + ifobj[i]['Event'])

            if i[0:3] == 'Set':
                self._manage_var_val_(ifobj[i]['Variable'], ifobj[i]['Value'], level)

    def _manage_json_event_(self, event, name, level):
        #
        # Manage events
        #
        for e in event['Events']:
            if isinstance(e, dict) and 'Event' in e:
                self._folder_.add_resource(consts.EVENT_PREFIX + e['Event'])
                # print(consts.EVENT_PREFIX + e['Event'])

    def _manage_json_capture_(self, capobj, name, level):
        #
        # Manage Capture
        #
        self._manage_var_val_(capobj['VariableName'], 0, level)

    def get_folder(self):
        #
        # Returns one folder object at the time from the list created when the fetcher was initiated
        #
        if self._indx_in_list_ < len(self._folders_list_):
            self._indx_in_list_ = self._indx_in_list_ + 1
            return (self._folders_list_[self._indx_in_list_ - 1])
        else:
            return None
